{
  lib,
  stdenv,
  fetchurl,
  makeShellWrapper,
  updateAutotoolsGnuConfigScriptsHook,
  runtimeShellPackage,
  # Tests
  gzip,
  less,
  perl,
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

stdenv.mkDerivation (finalAttrs: {
  pname = "gzip";
  version = "1.14";

  src = fetchurl {
    url = "mirror://gnu/gzip/${finalAttrs.pname}-${finalAttrs.version}.tar.xz";
    hash = "sha256-Aae4gb0iC/32Ffl7hxj4C9/T9q3ThbmT3Pbv0U6MCsY=";
  };

  outputs = [
    "out"
    "man"
    "info"
  ];

  enableParallelBuilding = true;

  nativeBuildInputs = [
    updateAutotoolsGnuConfigScriptsHook
    makeShellWrapper
  ];
  buildInputs = [ runtimeShellPackage ];

  makeFlags = [
    "SHELL=/bin/sh"
    "GREP=grep"
    # gzip 1.12 doesn't build `zless` unless it can find `less`, but we
    # can avoid having `less` as a build input if we just override these.
    "ZLESS_MAN=zless.1"
    "ZLESS_PROG=zless"
  ];

  nativeCheckInputs = [
    less
    perl
  ];
  doCheck = false;

  # Many gzip executables are shell scripts that depend upon other gzip
  # executables being in $PATH.  Rather than try to re-write all the
  # internal cross-references, just add $out/bin to PATH at the top of
  # all the executables that are shell scripts.
  preFixup = ''
    sed -i '1{;/#!\/bin\/sh/aPATH="'$out'/bin:$PATH"
    }' $out/bin/*
  ''
  # avoid wrapping the actual executable on cygwin because changing the
  # extension will break dll linking
  + lib.optionalString stdenv.hostPlatform.isCygwin ''
    mv $out/bin/{,.}gzip.exe
    ln -s .gzip.exe $out/bin/gzip
  ''
  # run gzip with "-n" when $GZIP_NO_TIMESTAMPS (set by stdenv's setup.sh) is set to stop gzip from adding timestamps
  # to archive headers: https://github.com/NixOS/nixpkgs/issues/86348
  # if changing so that there's no longer a .gzip-wrapped then update copy in make-bootstrap-tools.nix
  + ''
    wrapProgram $out/bin/gzip \
      --add-flags "\''${GZIP_NO_TIMESTAMPS:+-n}"
  '';

  passthru.tests.makecheck = gzip.overrideAttrs { doCheck = true; };

  meta = {
    homepage = "https://www.gnu.org/software/gzip/";
    description = "GNU zip compression program";
    longDescription = ''
      gzip (GNU zip) is a popular data compression program written by
      Jean-loup Gailly for the GNU project.  Mark Adler wrote the
      decompression part.

      We developed this program as a replacement for compress because of
      the Unisys and IBM patents covering the LZW algorithm used by
      compress.  These patents made it impossible for us to use compress,
      and we needed a replacement.  The superior compression ratio of gzip
      is just a bonus.
    '';
    platforms = lib.platforms.all;
    license = lib.licenses.gpl3Plus;
    mainProgram = "gzip";
    maintainers = [ lib.maintainers.mdaniels5757 ];
  };
})
