{
  stdenv,
  lib,
  fetchurl,
  libiconv,
  bashNonInteractive,
  updateAutotoolsGnuConfigScriptsHook,
  gnulib,
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

stdenv.mkDerivation rec {
  pname = "gettext";
  version = "0.25.1";

  src = fetchurl {
    url = "mirror://gnu/gettext/${pname}-${version}.tar.gz";
    hash = "sha256-dG+VXULXHrac52OGnLkmgvCaQGZSjQGLbKej9ICJoIU=";
  };
  patches = [
    ./absolute-paths.diff
    # fix reproducibile output, in particular in the grub2 build
    # https://savannah.gnu.org/bugs/index.php?59658
    ./0001-msginit-Do-not-use-POT-Creation-Date.patch
    ./memory-safety.patch
  ];

  outputs = [
    "out"
    "man"
    "doc"
    "info"
  ];

  configureFlags = [
    "--disable-csharp"
  ]
  ++ lib.optionals (stdenv.hostPlatform != stdenv.buildPlatform) [
    # On cross building, gettext supposes that the wchar.h from libc
    # does not fulfill gettext needs, so it tries to work with its
    # own wchar.h file, which does not cope well with the system's
    # wchar.h and stddef.h (gcc-4.3 - glibc-2.9)
    "gl_cv_func_wcwidth_works=yes"
  ];

  postPatch = ''
    # Older versions of gettext come with a copy of `extern-inline.m4` that is not compatible with clang 18.
    # When a project uses gettext + autoreconfPhase, autoreconfPhase will invoke `autopoint -f`, which will
    # replace whatever (probably compatible) version of `extern-inline.m4` with one that probalby won’t work
    # because `autopoint` will copy the autoconf macros from the project’s required version of gettext.
    # Fixing this requires replacing all the older copies of the problematic file with a new one.
    #
    # This is ugly, but it avoids requiring workarounds in every package using gettext and autoreconfPhase.
    declare -a oldFiles=($(tar tf gettext-tools/misc/archive.dir.tar | grep '^gettext-0\.[19].*/extern-inline.m4'))
    oldFilesDir=$(mktemp -d)
    for oldFile in "''${oldFiles[@]}"; do
      mkdir -p "$oldFilesDir/$(dirname "$oldFile")"
      cp -a gettext-tools/gnulib-m4/extern-inline.m4 "$oldFilesDir/$oldFile"
    done
    tar uf gettext-tools/misc/archive.dir.tar --owner=0 --group=0 --numeric-owner -C "$oldFilesDir" "''${oldFiles[@]}"

    substituteAllInPlace gettext-runtime/src/gettext.sh.in
    substituteInPlace gettext-tools/projects/KDE/trigger --replace "/bin/pwd" pwd
    substituteInPlace gettext-tools/projects/GNOME/trigger --replace "/bin/pwd" pwd
    substituteInPlace gettext-tools/src/project-id --replace "/bin/pwd" pwd
  ''
  + lib.optionalString stdenv.hostPlatform.isMinGW ''
    sed -i "s/@GNULIB_CLOSE@/1/" */*/unistd.in.h
  ''
  + lib.optionalString stdenv.hostPlatform.isCygwin ''
    for gnulib in \
      ./libtextstyle/lib \
      ./gettext-tools/libgettextpo \
      ./gettext-tools/gnulib-lib \
      ./gettext-runtime/libasprintf/gnulib-lib \
      ./gettext-runtime/intl/gnulib-lib \
      ./gettext-runtime/gnulib-lib
    do
      cd "$gnulib"
      patch -p2 < ${gnulib.patches.memcpy-fix}
      cd -
    done
  '';

  strictDeps = true;
  nativeBuildInputs = [
    updateAutotoolsGnuConfigScriptsHook
  ];
  buildInputs =
    lib.optionals (!stdenv.hostPlatform.isMinGW) [
      bashNonInteractive
    ]
    ++ lib.optionals (!stdenv.hostPlatform.isLinux) [
      # HACK, see #10874 (and 14664)
      libiconv
    ];

  setupHooks = [
    ../../../build-support/setup-hooks/role.bash
    ./gettext-setup-hook.sh
  ];
  env = {
    gettextNeedsLdflags = stdenv.hostPlatform.libc != "glibc" && !stdenv.hostPlatform.isMusl;
  }
  // lib.optionalAttrs stdenv.hostPlatform.isDarwin {
    # macOS iconv implementation is slightly broken since Sonoma
    # https://github.com/Homebrew/homebrew-core/pull/199639
    # https://savannah.gnu.org/bugs/index.php?66541
    am_cv_func_iconv_works = "yes";
  }
  // lib.optionalAttrs stdenv.hostPlatform.isSunOS {
    LDFLAGS = "-lm -lmd -lmp -luutil -lnvpair -lnsl -lidmap -lavl -lsec";
  };

  enableParallelBuilding = true;
  enableParallelChecking = false; # fails sometimes

  meta = {
    description = "Well integrated set of translation tools and documentation";

    longDescription = ''
      Usually, programs are written and documented in English, and use
      English at execution time for interacting with users.  Using a common
      language is quite handy for communication between developers,
      maintainers and users from all countries.  On the other hand, most
      people are less comfortable with English than with their own native
      language, and would rather be using their mother tongue for day to
      day's work, as far as possible.  Many would simply love seeing their
      computer screen showing a lot less of English, and far more of their
      own language.

      GNU `gettext` is an important step for the GNU Translation Project, as
      it is an asset on which we may build many other steps. This package
      offers to programmers, translators, and even users, a well integrated
      set of tools and documentation. Specifically, the GNU `gettext`
      utilities are a set of tools that provides a framework to help other
      GNU packages produce multi-lingual messages.
    '';

    homepage = "https://www.gnu.org/software/gettext/";

    maintainers = with lib.maintainers; [ zimbatm ];
    license = lib.licenses.gpl2Plus;
    platforms = lib.platforms.all;
  };
}

// lib.optionalAttrs stdenv.hostPlatform.isDarwin {
  makeFlags = [ "CFLAGS=-D_FORTIFY_SOURCE=0" ];
}
