{
  stdenv,
  lib,
  fetchFromGitHub,
  fetchpatch2,
  fetchurl,
}:

# This file is responsible for fetching the sage source and adding necessary patches.
# It does not actually build anything, it just copies the patched sources to $out.
# This is done because multiple derivations rely on these sources and they should
# all get the same sources with the same patches applied.

stdenv.mkDerivation rec {
  version = "10.7";
  pname = "sage-src";

  src = fetchFromGitHub {
    owner = "sagemath";
    repo = "sage";
    rev = version;
    hash = "sha256-nYlBmKQ9TD5EAVvNwo8YzqAd5IUCpTU3kBTqUH21IxQ=";
  };

  # contains essential files (e.g., setup.cfg) generated by the bootstrap script.
  # TODO: investigate https://github.com/sagemath/sage/pull/35950
  configure-src = fetchurl {
    # the hash below is the tagged commit's _parent_. it can also be found by looking for
    # the "configure" asset at https://github.com/sagemath/sage/releases/tag/${version}
    url = "mirror://sageupstream/configure/configure-858268b40010e5ed6da13488ad0f52cda4d1f70e.tar.gz";
    hash = "sha256-TsVX+wUWr+keCXmGQp1OHGXgNc7luajyGxfTwduSEtc=";
  };

  # Patches needed because of particularities of nix or the way this is packaged.
  # The goal is to upstream all of them and get rid of this list.
  nixPatches = [
    # Parallelize docubuild using subprocesses, fixing an isolation issue. See
    # https://groups.google.com/forum/#!topic/sage-packaging/YGOm8tkADrE
    ./patches/sphinx-docbuild-subprocesses.patch

    # After updating smypow to (https://github.com/sagemath/sage/issues/3360)
    # we can now set the cache dir to be within the .sage directory. This is
    # not strictly necessary, but keeps us from littering in the user's HOME.
    ./patches/sympow-cache.patch
  ]
  ++ lib.optionals (stdenv.cc.isClang) [
    # https://github.com/NixOS/nixpkgs/pull/264126
    # Dead links in python sysconfig cause LLVM linker warnings, leading to cython doctest failures.
    ./patches/silence-linker.patch

    # Stack overflows during doctests; this does not change functionality.
    ./patches/disable-singular-doctest.patch
  ];

  # Since sage unfortunately does not release bugfix releases, packagers must
  # fix those bugs themselves. This is for critical bugfixes, where "critical"
  # == "causes (transient) doctest failures / somebody complained".
  bugfixPatches = [
    # compile libs/gap/element.pyx with -O1
    # a more conservative version of https://github.com/sagemath/sage/pull/37951
    ./patches/gap-element-crash.patch

    # https://github.com/sagemath/sage/pull/40895, landed in 10.8.beta6
    (fetchpatch2 {
      name = "doctest-absolute-path.patch";
      url = "https://github.com/sagemath/sage/commit/3a5904d43f552bf63ed1eed9154f87b1f0de53fb.patch?full_index=1";
      hash = "sha256-rp+9d8Y6kifWzufE07GWU68txPn//w7uMn4LcpITaBs=";
    })
  ];

  # Patches needed because of package updates. We could just pin the versions of
  # dependencies, but that would lead to rebuilds, confusion and the burdons of
  # maintaining multiple versions of dependencies. Instead we try to make sage
  # compatible with never dependency versions when possible. All these changes
  # should come from or be proposed to upstream. This list will probably never
  # be empty since dependencies update all the time.
  packageUpgradePatches = [
    # https://github.com/sagemath/sage/pull/40919, landed in 10.8.beta8
    (fetchpatch2 {
      name = "gap-4.15.1-update.patch";
      url = "https://github.com/sagemath/sage/commit/54d4ddb132cc71ef26b4db1f48afd6736d41cc63.patch?full_index=1";
      hash = "sha256-PZyOXRsgcsPvgceGGZXet5URJgWiIlCfFx8tvwpLk5A=";
      excludes = [ "src/doc/zh/constructions/rep_theory.rst" ];
    })

    # https://github.com/sagemath/sage/pull/41141, landed in 10.8.beta9
    (fetchpatch2 {
      name = "ipython-9_7_0-unicode_to_str.patch";
      url = "https://github.com/sagemath/sage/commit/fa00696112fde95e0c4241ad6063936200ce6f68.patch?full_index=1";
      hash = "sha256-eFDpNu/2gcQATELmQ7/VzXI35xIzaVrD8bhhJ57e2gc=";
    })

    # https://github.com/sagemath/sage/pull/41233, landed in 10.8.rc0
    (fetchpatch2 {
      name = "flint-3.4-update.patch";
      url = "https://github.com/sagemath/sage/commit/9bd5304f1a222e215d7006a04854ff66616748bf.patch?full_index=1";
      hash = "sha256-7bgouXV3pM20IX5PM24ZpJWOBlZjQksGtjopgouEbyg=";
    })
  ];

  patches = nixPatches ++ bugfixPatches ++ packageUpgradePatches;

  # do not create .orig backup files if patch applies with fuzz
  patchFlags = [
    "--no-backup-if-mismatch"
    "-p1"
  ];

  # harmless broken symlinks to (not) generated files used by sage-the-distro
  dontCheckForBrokenSymlinks = true;

  postPatch = ''
    # Make sure sage can at least be imported without setting any environment
    # variables. It won't be close to feature complete though.
    sed -i \
      "s|var(\"SAGE_ROOT\".*|var(\"SAGE_ROOT\", \"$out\")|" \
      src/sage/env.py

    # sage --docbuild unsets JUPYTER_PATH, which breaks our docbuilding
    # https://trac.sagemath.org/ticket/33650#comment:32
    sed -i "/export JUPYTER_PATH/d" src/bin/sage
  '';

  buildPhase = "# do nothing";

  installPhase = ''
    cp -r . "$out"
    tar xzf ${configure-src} -C "$out"
    rm "$out/configure"
  '';
}
