{
  stdenv,
  lib,
  autoPatchelfHook,
  buildFHSEnv,
  cairo,
  dpkg,
  fetchurl,
  gcc-unwrapped,
  glib,
  glibc,
  gtk2-x11,
  libGL,
  libpulseaudio,
  libSM,
  libXxf86vm,
  libX11,
  openssl_1_1,
  pango,
  SDL2,
  wrapGAppsHook3,
  xdg-utils,
  xorg,
  xorg_sys_opengl,
  zlib,
}:
let

  runescape = stdenv.mkDerivation rec {
    pname = "runescape-launcher";
    version = "2.2.11";

    # Packages: https://content.runescape.com/downloads/ubuntu/dists/trusty/non-free/binary-amd64/Packages
    # upstream is https://content.runescape.com/downloads/ubuntu/pool/non-free/r/${pname}/${pname}_${version}_amd64.deb
    src = fetchurl {
      url = "https://archive.org/download/${pname}_${version}_amd64/${pname}_${version}_amd64.deb";
      sha256 = "0dyilgbsr28zqpf711wygg706vn7sqxklnsnbghwkxfzzjppz2xw";
    };

    nativeBuildInputs = [
      autoPatchelfHook
      dpkg
      wrapGAppsHook3
    ];

    buildInputs = [
      cairo
      gcc-unwrapped
      glib
      glibc
      gtk2-x11
      libSM
      libXxf86vm
      libX11
      openssl_1_1
      pango
      zlib
    ];

    runtimeDependencies = [
      libGL
      libpulseaudio
      SDL2
      openssl_1_1
      xdg-utils # The launcher uses `xdg-open` to open HTTP URLs in the user's browser
      xorg_sys_opengl
      zlib
    ];

    dontUnpack = true;

    preBuild = ''
      export DH_VERBOSE=1
    '';

    envVarsWithXmodifiers = ''
      export MESA_GLSL_CACHE_DIR=~/Jagex
      export GDK_SCALE=2
      unset XMODIFIERS
    '';

    installPhase = ''
      mkdir -p $out/bin $out/share
      dpkg -x $src $out

      patchShebangs $out/usr/bin/runescape-launcher
      substituteInPlace $out/usr/bin/runescape-launcher \
        --replace "unset XMODIFIERS" "$envVarsWithXmodifiers" \
        --replace "/usr/share/games/runescape-launcher/runescape" "$out/share/games/runescape-launcher/runescape"

      cp -r $out/usr/bin $out/
      cp -r $out/usr/share $out/

      rm -r $out/usr
    '';

    meta = {
      description = "Launcher for RuneScape 3, the current main RuneScape";
      homepage = "https://www.runescape.com/";
      sourceProvenance = with lib.sourceTypes; [ binaryNativeCode ];
      license = lib.licenses.unfree;
      maintainers = [ ];
      platforms = [ "x86_64-linux" ];
    };
  };

in

/*
  * We can patch the runescape launcher, but it downloads a client at runtime and checks it for changes.
  * For that we need use a buildFHSEnv.
  * FHS simulates a classic linux shell
*/
buildFHSEnv {
  pname = "RuneScape";
  inherit (runescape) version;
  targetPkgs = pkgs: [
    runescape
    cairo
    dpkg
    gcc-unwrapped
    glib
    glibc
    gtk2-x11
    libGL
    libpulseaudio
    libSM
    libXxf86vm
    libX11
    openssl_1_1
    pango
    SDL2
    xdg-utils
    xorg.libX11
    xorg_sys_opengl
    zlib
  ];
  multiPkgs = pkgs: [ libGL ];
  runScript = "runescape-launcher";
  extraInstallCommands = ''
    mkdir -p "$out/share/applications"
    cp ${runescape}/share/applications/runescape-launcher.desktop "$out/share/applications"
    cp -r ${runescape}/share/icons "$out/share/icons"
    substituteInPlace "$out/share/applications/runescape-launcher.desktop" \
      --replace "/usr/bin/runescape-launcher" "RuneScape"
  '';

  meta = {
    description = "RuneScape Game Client (NXT) - Launcher for RuneScape 3";
    homepage = "https://www.runescape.com/";
    license = lib.licenses.unfree;
    maintainers = [ ];
    platforms = [ "x86_64-linux" ];
  };
}
