{
  lib,
  stdenv,
  fetchFromGitHub,
  cmake,
  doxygen,
  gettext,
  graphviz,
  libxslt,
  removeReferencesTo,
  libiconv,
  brotli,
  expat,
  inih,
  zlib,
  libxml2,
  python3,
  which,
}:

stdenv.mkDerivation rec {
  pname = "exiv2";
  version = "0.28.7";

  outputs = [
    "out"
    "lib"
    "dev"
    "doc"
    "man"
  ];

  src = fetchFromGitHub {
    owner = "exiv2";
    repo = "exiv2";
    tag = "v${version}";
    hash = "sha256-a7nPjDjTcwsQeypARvy2rRsv9jpasSSxSyCTLWNDDtA=";
  };

  nativeBuildInputs = [
    cmake
    doxygen
    gettext
    graphviz
    libxslt
    removeReferencesTo
  ];

  buildInputs = lib.optionals stdenv.hostPlatform.isDarwin [
    libiconv
  ];

  propagatedBuildInputs = [
    brotli
    expat
    inih
    zlib
  ];

  nativeCheckInputs = [
    libxml2.bin
    python3
    which
  ];

  cmakeFlags = [
    "-DEXIV2_ENABLE_NLS=ON"
    "-DEXIV2_BUILD_DOC=ON"
    "-DEXIV2_ENABLE_BMFF=ON"
  ];

  buildFlags = [
    "all"
    "doc"
  ];

  doCheck = true;

  preCheck = ''
    patchShebangs ../test/
    mkdir ../test/tmp

    # template.exv_test (test_regression_allfiles.TestAllFiles.template.exv_test) ... ERROR
    substituteInPlace ../tests/regression_tests/test_regression_allfiles.py \
      --replace-fail '"issue_2403_poc.exv",' '"issue_2403_poc.exv", "template.exv",'
  ''
  + lib.optionalString stdenv.hostPlatform.isAarch32 ''
    # Fix tests on arm
    # https://github.com/Exiv2/exiv2/issues/933
    rm -f ../tests/bugfixes/github/test_CVE_2018_12265.py
  ''
  + lib.optionalString stdenv.hostPlatform.isDarwin ''
    export DYLD_LIBRARY_PATH=$DYLD_LIBRARY_PATH''${DYLD_LIBRARY_PATH:+:}$PWD/lib
    export LC_ALL=C

    # disable tests that requires loopback networking
    substituteInPlace ../tests/bash_tests/testcases.py \
      --replace-fail "def io_test(self):" "def io_disabled(self):"
  '';

  preFixup = ''
    remove-references-to -t ${stdenv.cc.cc} $lib/lib/*.so.*.*.* $out/bin/exiv2
  '';

  disallowedReferences = [ stdenv.cc.cc ];

  # causes redefinition of _FORTIFY_SOURCE
  hardeningDisable = [ "fortify3" ];

  meta = {
    homepage = "https://exiv2.org";
    description = "Library and command-line utility to manage image metadata";
    mainProgram = "exiv2";
    platforms = lib.platforms.all;
    license = lib.licenses.gpl2Plus;
    maintainers = with lib.maintainers; [ wegank ];
  };
}
