{
  lib,
  python3,
  fetchPypi,
  fetchFromGitHub,
  gitMinimal,
  stdenv,
  installShellFiles,
  nix-update-script,
  versionCheckHook,
  writableTmpDirAsHomeHook,
}:
let
  # commitizen 4.9.1 is not compatible with version 3.0.52 of prompt-toolkit
  python = python3.override {
    packageOverrides = self: super: {
      prompt-toolkit = super.prompt-toolkit.overridePythonAttrs (oldAttrs: rec {
        version = "3.0.51";
        pname = "prompt_toolkit";
        src = fetchPypi {
          inherit pname version;
          hash = "sha256-kxoWLjsn/JDIbxtIux+yxSjCdhR15XycBt4TMRx7VO0=";
        };
      });
    };
  };
  python3Packages = python.pkgs;
in
python3Packages.buildPythonPackage rec {
  pname = "commitizen";
  version = "4.10.1";
  pyproject = true;

  src = fetchFromGitHub {
    owner = "commitizen-tools";
    repo = "commitizen";
    tag = "v${version}";
    hash = "sha256-B4V2UPTEXQNASrwGRZbfFOqPuBIFzBM39a5rAC+Hk5Q=";
  };

  pythonRelaxDeps = [
    "argcomplete"
    "decli"
    "termcolor"
  ];

  build-system = with python3Packages; [ poetry-core ];

  nativeBuildInputs = [ installShellFiles ];

  dependencies = with python3Packages; [
    argcomplete
    charset-normalizer
    colorama
    decli
    deprecated
    importlib-metadata
    jinja2
    packaging
    prompt-toolkit
    pyyaml
    questionary
    termcolor
    tomlkit
  ];

  nativeCheckInputs = [
    gitMinimal
    versionCheckHook
    writableTmpDirAsHomeHook
  ]
  ++ (with python3Packages; [
    argcomplete
    py
    pytest-freezer
    pytest-mock
    pytest-regressions
    pytest7CheckHook
  ]);

  versionCheckProgramArg = "version";

  pythonImportsCheck = [ "commitizen" ];

  # The tests require a functional git installation
  preCheck = ''
    git config --global user.name "Nix Builder"
    git config --global user.email "nix-builder@nixos.org"
    git init .
  '';

  # NB: These tests require complex GnuPG setup
  disabledTests = [
    "test_bump_minor_increment_signed"
    "test_bump_minor_increment_signed_config_file"
    "test_bump_on_git_with_hooks_no_verify_enabled"
    "test_bump_on_git_with_hooks_no_verify_disabled"
    "test_bump_pre_commit_changelog"
    "test_bump_pre_commit_changelog_fails_always"
    "test_get_commits_with_signature"
    # fatal: not a git repository (or any of the parent directories): .git
    "test_commitizen_debug_excepthook"
  ];

  postInstall =
    let
      register-python-argcomplete = lib.getExe' python3Packages.argcomplete "register-python-argcomplete";
    in
    lib.optionalString (stdenv.buildPlatform.canExecute stdenv.hostPlatform) ''
      installShellCompletion --cmd cz \
        --bash <(${register-python-argcomplete} --shell bash $out/bin/cz) \
        --zsh <(${register-python-argcomplete} --shell zsh $out/bin/cz) \
        --fish <(${register-python-argcomplete} --shell fish $out/bin/cz)
    '';

  passthru = {
    updateScript = nix-update-script { };
  };

  meta = {
    description = "Tool to create committing rules for projects, auto bump versions, and generate changelogs";
    homepage = "https://github.com/commitizen-tools/commitizen";
    changelog = "https://github.com/commitizen-tools/commitizen/blob/${src.tag}/CHANGELOG.md";
    license = lib.licenses.mit;
    mainProgram = "cz";
    maintainers = with lib.maintainers; [
      lovesegfault
      anthonyroussel
    ];
  };
}
