#!/usr/bin/env python
# format:
# $ nix run nixpkgs#python3Packages.ruff -- updater.py
# type-check:
# $ nix run nixpkgs#python3Packages.mypy -- updater.py
# linted:
# $ nix run nixpkgs#python3Packages.flake8 -- --ignore E501,E265,E402 updater.py

import csv
import inspect
import logging
import os
import re
import shutil
import subprocess
import tempfile
import textwrap
from dataclasses import dataclass
from multiprocessing.dummy import Pool
from pathlib import Path

import nixpkgs_plugin_update  # type: ignore
from nixpkgs_plugin_update import FetchConfig, Redirects, commit, retry, update_plugins


class ColoredFormatter(logging.Formatter):
    # Define color codes
    COLORS = {
        "DEBUG": "\033[94m",  # Blue
        "INFO": "\033[92m",  # Green
        "WARNING": "\033[93m",  # Yellow
        "ERROR": "\033[91m",  # Red
        "CRITICAL": "\033[95m",  # Magenta
    }
    RESET = "\033[0m"

    def format(self, record):
        log_color = self.COLORS.get(record.levelname, self.RESET)
        record.msg = f"{log_color}{record.msg}{self.RESET}"
        return super().format(record)


handler = logging.StreamHandler()
handler.setFormatter(ColoredFormatter("%(levelname)s: %(message)s"))
log = logging.getLogger()
log.handlers = [handler]

ROOT = Path(os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))).parent.parent  # type: ignore

PKG_LIST = "maintainers/scripts/luarocks-packages.csv"
TMP_FILE = "$(mktemp)"
GENERATED_NIXFILE = "pkgs/development/lua-modules/generated-packages.nix"

HEADER = """/*
  {GENERATED_NIXFILE} is an auto-generated file -- DO NOT EDIT!
  Regenerate it with: nix run nixpkgs#luarocks-packages-updater
  You can customize the generated packages in pkgs/development/lua-modules/overrides.nix
*/
""".format(GENERATED_NIXFILE=GENERATED_NIXFILE)

FOOTER = (
    textwrap.dedent("""
}
# GENERATED - do not edit this file
""").strip()
    + "\n"
)


@dataclass
class LuaPlugin:
    name: str
    """Name of the plugin, as seen on luarocks.org"""
    rockspec: str
    """Full URI towards the rockspec"""
    ref: str | None
    """git reference (branch name/tag)"""
    version: str | None
    """Set it to pin a package """
    server: str | None
    """luarocks.org registers packages under different manifests.
    Its value can be 'http://luarocks.org/dev'
    """
    luaversion: str | None
    """lua version if a package is available only for a specific lua version"""
    maintainers: str | None
    """Optional string listing maintainers separated by spaces"""

    @property
    def normalized_name(self) -> str:
        return self.name.replace(".", "-")


def extract_version(nix_expr: str) -> str | None:
    match = re.search(r'version\s*=\s*"([^"]+)"', nix_expr)
    if match:
        return match.group(1)
    return None


def extract_rev(nix_expr: str) -> str | None:
    match = re.search(r'rev\s*=\s*"([^"]+)"', nix_expr)
    if match:
        return match.group(1)
    return None


# rename Editor to LangUpdate/ EcosystemUpdater
class LuaEditor(nixpkgs_plugin_update.Editor):
    def create_parser(self):
        parser = super().create_parser()
        parser.set_defaults(proc=1)
        return parser

    def get_current_plugins(self, _config: FetchConfig, _nixpkgs: str):
        return []

    def load_plugin_spec(self, _config: FetchConfig, input_file) -> list[LuaPlugin]:
        luaPackages = []
        csvfilename = input_file
        log.info("Loading package descriptions from %s", csvfilename)

        with open(csvfilename, newline="") as csvfile:
            reader = csv.DictReader(
                csvfile,
            )
            for row in reader:
                # name,server,version,luaversion,maintainers
                plugin = LuaPlugin(**row)
                luaPackages.append(plugin)
        return luaPackages

    def update(self, args):
        update_plugins(self, args)

    def generate_nix(self, results: list[tuple[LuaPlugin, str]], outfilename: str):
        with tempfile.NamedTemporaryFile("w+") as f:
            f.write(HEADER)
            header2 = textwrap.dedent(
                """
            {
              stdenv,
              lib,
              fetchurl,
              fetchgit,
              callPackage,
              ...
            }:
            final: prev: {
            """
            )
            f.write(header2)
            for plugin, nix_expr in results:
                f.write(f"  {plugin.normalized_name} = {nix_expr}")
            f.write(FOOTER)
            f.flush()

            # if everything went fine, move the generated file to its destination
            # using copy since move doesn't work across disks
            shutil.copy(f.name, outfilename)

        print(f"updated {outfilename}")

        # Format the generated file with nix fmt
        subprocess.run(["nix", "fmt", outfilename], check=True)

    @property
    def attr_path(self):
        return "luaPackages"

    def parse_generated_nix(self, input_file: str) -> dict[str, str]:
        plugins: dict[str, str] = {}
        if not os.path.exists(input_file):
            return plugins

        with open(input_file, "r") as f:
            content = f.read()

        start_marker = "final: prev: {"
        start_idx = content.find(start_marker)
        if start_idx == -1:
            log.warning("Could not find start marker in generated file")
            return plugins
        start_idx += len(start_marker)

        # We assume the file ends with the footer or at least a closing brace
        lines = content[start_idx:].splitlines()

        current_name = None
        current_lines = []

        # Regex to match start of a plugin definition:   name = callPackage (
        start_pattern = re.compile(r"^\s+([\w\.\-]+)\s+=\s+callPackage\s+\(")
        # Regex to match end of a plugin definition:   ) { };
        end_pattern = re.compile(r"^\s+\)\s+\{\s+\};$")

        for line in lines:
            if current_name is None:
                match = start_pattern.match(line)
                if match:
                    current_name = match.group(1)
                    # We need to keep the RHS of the assignment
                    # line is "  name = callPackage ("
                    # We want "callPackage ("
                    rhs = line.split("=", 1)[1].strip()
                    current_lines = [rhs]
            else:
                current_lines.append(line)
                if end_pattern.match(line):
                    plugins[current_name] = "\n".join(current_lines) + "\n\n"
                    current_name = None
                    current_lines = []

        return plugins

    def add(self, args):
        if not args.add_plugins:
            return

        fieldnames = ["name", "rockspec", "ref", "server", "version", "luaversion", "maintainers"]
        fetch_config = FetchConfig(args.proc, args.github_token)

        for plugin_name in args.add_plugins:
            log.info(f"Adding {plugin_name} to CSV")

            existing_entries = []
            with open(self.default_in, "r", newline="") as csvfile:
                reader = csv.DictReader(csvfile)
                existing_entries = list(reader)

            new_entry = {
                "name": plugin_name,
                "rockspec": "",
                "ref": "",
                "server": "",
                "version": "",
                "luaversion": "",
                "maintainers": "",
            }
            existing_entries.append(new_entry)

            existing_entries.sort(key=lambda x: x["name"].lower())

            with open(self.default_in, "w", newline="") as csvfile:
                writer = csv.DictWriter(csvfile, fieldnames=fieldnames, lineterminator="\n")
                writer.writeheader()
                writer.writerows(existing_entries)

            update = self.get_update(str(self.default_in), str(args.outfile), fetch_config, to_update=[plugin_name])
            redirects, updated_plugins = update()

            if not args.no_commit and updated_plugins:
                for name, old_ver, new_ver in updated_plugins:
                    if old_ver == "init":
                        msg = f"{self.attr_path}.{name}: init at {new_ver}"
                    else:
                        msg = f"{self.attr_path}.{name}: {old_ver} -> {new_ver}"

                    commit(self.nixpkgs_repo, msg, [args.outfile, self.default_in])

    def get_update(
        self,
        input_file: str,
        output_file: str,
        config: FetchConfig,
        to_update: list[str] | None,
    ):
        def update() -> tuple[Redirects, list[tuple[str, str, str]]]:
            all_plugin_specs = self.load_plugin_spec(config, input_file)
            sorted_all_specs = sorted(all_plugin_specs, key=lambda v: v.name.lower())

            specs_to_process = sorted_all_specs
            if to_update:
                specs_to_process = [
                    p for p in sorted_all_specs if p.normalized_name in to_update or p.name in to_update
                ]

            # Load existing plugins to preserve them if update fails or skipped
            existing_plugins = self.parse_generated_nix(output_file)

            old_versions = {}
            for name, expr in existing_plugins.items():
                v = extract_version(expr)
                if v:
                    r = extract_rev(expr)
                    old_versions[name] = f"{v}-{r}" if r else v

            try:
                pool = Pool(processes=config.proc)
                results = pool.map(generate_pkg_nix, specs_to_process)
            finally:
                pass

            results_map = {}
            for plug, nix_expr, error in results:
                results_map[plug.normalized_name] = (nix_expr, error)

            successful_results = []
            errors = []
            updated_plugins: list[tuple[str, str, str]] = []

            # Iterate over ALL specs to rebuild the full file
            for plug in sorted_all_specs:
                final_expr = None

                # Check if this plugin was processed in this run
                if plug.normalized_name in results_map:
                    nix_expr, error = results_map[plug.normalized_name]

                    if nix_expr:
                        final_expr = nix_expr
                    elif plug.normalized_name in existing_plugins:
                        # Update failed, fallback to existing
                        log.warning(f"Update failed for {plug.name}, keeping existing version. Error: {error}")
                        final_expr = existing_plugins[plug.normalized_name]
                        errors.append((plug, error))
                    else:
                        # Update failed with no fallback
                        log.error(f"Update failed for {plug.name} and no existing version found. Error: {error}")
                        errors.append((plug, error))
                else:
                    # Not processed this run, use existing if available
                    final_expr = existing_plugins.get(plug.normalized_name)

                if final_expr:
                    successful_results.append((plug, final_expr))
                    track_version_change(plug, final_expr, old_versions, updated_plugins)

            self.generate_nix(successful_results, output_file)

            if errors:
                log.error("The following plugins failed to update:")
                for plug, error in errors:
                    log.error("%s: %s", plug.name, error)

            redirects: Redirects = {}
            return redirects, updated_plugins

        return update

    def rewrite_input(self, _config: FetchConfig, _input_file: str, *args, **kwargs):
        # vim plugin reads the file before update but that shouldn't be our case
        # not implemented yet
        # fieldnames = ['name', 'server', 'version', 'luaversion', 'maintainers']
        # input_file = "toto.csv"
        # with open(input_file, newline='') as csvfile:
        #     writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
        #     writer.writeheader()
        #     for row in reader:
        #         # name,server,version,luaversion,maintainers
        #         plugin = LuaPlugin(**row)
        #         luaPackages.append(plugin)
        pass


def track_version_change(
    plug: LuaPlugin, nix_expr: str, old_versions: dict[str, str], updated_plugins: list[tuple[str, str, str]]
) -> None:
    """Track version changes for a plugin."""
    v = extract_version(nix_expr)
    if not v:
        return

    r = extract_rev(nix_expr)
    new_ver = f"{v}-{r}" if r else v
    old_ver = old_versions.get(plug.normalized_name)

    if old_ver is None:
        updated_plugins.append((plug.normalized_name, "init", new_ver))
    elif new_ver != old_ver:
        updated_plugins.append((plug.normalized_name, old_ver, new_ver))


@retry(subprocess.CalledProcessError, tries=3, delay=3, backoff=2)
def run_luarocks(cmd: list[str]) -> str:
    return subprocess.check_output(cmd, text=True)


def generate_pkg_nix(plug: LuaPlugin):
    """
    Generate nix expression for a luarocks package
    Our cache key associates "p.name-p.version" to its rockspec
    """
    log.debug("Generating nix expression for %s", plug.name)

    try:
        cmd = ["luarocks", "nix"]

        if plug.maintainers:
            cmd.append(f"--maintainers={plug.maintainers}")

        if plug.rockspec != "":
            if plug.ref or plug.version:
                msg = "'version' and 'ref' will be ignored as the rockspec is hardcoded for package %s" % plug.name
                log.warning(msg)

            log.debug("Updating from rockspec %s", plug.rockspec)
            cmd.append(plug.rockspec)

        # update the plugin from luarocks
        else:
            cmd.append(plug.name)
            if plug.version and plug.version != "src":
                cmd.append(plug.version)

        if plug.server != "src" and plug.server:
            cmd.append(f"--only-server={plug.server}")

        if plug.luaversion:
            cmd.append(f"--lua-version={plug.luaversion}")
            luaver = plug.luaversion.replace(".", "")
            if lua_dir := os.getenv(f"LUA_{luaver}"):
                cmd.append(f"--lua-dir={lua_dir}")

        log.debug("running %s", " ".join(cmd))

        output = run_luarocks(cmd)
        ## FIXME: luarocks nix command output isn't formatted properly
        output = "callPackage(\n" + output.strip() + ") {};\n\n"
        return (plug, output, None)
    except subprocess.CalledProcessError as e:
        log.error("Failed to generate nix expression for %s: %s", plug.name, e)
        return (plug, None, str(e))


def main():
    editor = LuaEditor(
        "lua",
        ROOT,
        "",
        default_in=Path(PKG_LIST),
        default_out=Path(GENERATED_NIXFILE),
    )

    editor.run()


if __name__ == "__main__":
    main()
