{
  pkgs,
  lib,
  config,
  ...
}:
let
  cfg = config.networking.getaddrinfo;

  formatTableEntries =
    tableName: table:
    if table == null then
      [ ]
    else
      lib.mapAttrsToList (cidr: val: "${tableName} ${cidr} ${toString val}") table;

  gaiConfText = lib.concatStringsSep "\n" (
    [
      "# Generated by NixOS module networking.getaddrinfo"
      "# Do not edit manually!"
      "reload ${lib.boolToYesNo cfg.reload}"
    ]
    ++ formatTableEntries "label" cfg.label
    ++ formatTableEntries "precedence" cfg.precedence
    ++ formatTableEntries "scopev4" cfg.scopev4
  );
in
{
  options.networking.getaddrinfo = {
    enable = lib.mkOption {
      type = lib.types.bool;
      default = pkgs.stdenv.hostPlatform.libc == "glibc";
      defaultText = lib.literalExpression ''
        pkgs.stdenv.hostPlatform.libc == "glibc"
      '';
      description = ''
        Enables custom address sorting configuration for {manpage}`getaddrinfo(3)` according to RFC 3484.

        This option generates a {file}`/etc/gai.conf` file to override the default address sorting tables,
        as described in {manpage}`gai.conf(5)`.

        This setting is only applicable when using the GNU C Library (glibc).
        It has no effect with other libc implementations.
      '';
    };

    reload = lib.mkOption {
      type = lib.types.bool;
      default = false;
      description = ''
        Determines whether a process should detect changes to the configuration file since it was last read.

        If enabled, the file is re-read automatically. This may cause issues in multithreaded applications
        and is generally discouraged.
      '';
    };

    label = lib.mkOption {
      type = lib.types.nullOr (lib.types.attrsOf lib.types.int);
      default = null;
      description = ''
        Adds entries to the label table, as described in section 2.1 of RFC 3484.

        If any label entries are provided, the glibc’s default label table is ignored.
      '';
      example = {
        "::/0" = 1;
        "2002::/16" = 2;
        "::/96" = 3;
        "::ffff:0:0/96" = 4;
        "fec0::/10" = 5;
        "fc00::/7" = 6;
        "2001:0::/32" = 7;
      };
    };

    precedence = lib.mkOption {
      type = lib.types.nullOr (lib.types.attrsOf lib.types.int);
      default = null;
      description = ''
        Similar to {option}`networking.getaddrinfo.label`, but this option
        defines entries for the precedence table instead.

        See sections 2.1 and 10.3 of RFC 3484 for details.

        Providing any value will disable the glibc's default precedence table.
      '';
      example = {
        "::1/128" = 50;
        "::/0" = 40;
        "2002::/16" = 30;
        "::/96" = 20;
        "::ffff:0:0/96" = 10;
      };
    };

    scopev4 = lib.mkOption {
      type = lib.types.nullOr (lib.types.attrsOf lib.types.int);
      default = null;
      description = ''
        Adds custom rules to the IPv4 scope table.

        By default, the scope IDs described in section 3.2 of RFC 6724 are used.

        Modifying these values is rarely necessary.
      '';
      example = {
        "::ffff:169.254.0.0/112" = 2;
        "::ffff:127.0.0.0/104" = 2;
        "::ffff:0.0.0.0/96" = 14;
      };
    };
  };

  config = lib.mkIf cfg.enable {
    environment.etc."gai.conf".text = gaiConfText;
  };

  meta.maintainers = with lib.maintainers; [ moraxyc ];
}
