{
  lib,
  python3Packages,
  fetchFromGitHub,
  procps,
  stdenv,
  versionCheckHook,
  addBinToPathHook,
}:

python3Packages.buildPythonApplication rec {
  pname = "multiqc";
  version = "1.30";
  pyproject = true;

  # Two data sources. One for the code, another for the test data
  srcs = [
    (fetchFromGitHub {
      name = "multiqc";
      owner = "MultiQC";
      repo = "MultiQC";
      tag = "v${version}";
      hash = "sha256-TR5YFoWj97gpsykIzc1lqtYVePsVLRIT0HXw+VPJ7o4=";
    })
    (fetchFromGitHub {
      owner = "MultiQC";
      repo = "test-data";
      rev = "cc9f853e7892eb537e91505e0e847ff63669138d";
      hash = "sha256-/MiNnPayG6wpF2S09ENYTlEBF7Km4aH1RjdGOfMgZcA=";
      name = "test-data";
    })
  ];

  # Multiqc cannot remove temporary directories in some case.
  # Default is 10 retries, lower it to 2
  postPatch = ''
    substituteInPlace multiqc/utils/util_functions.py \
      --replace-fail \
        "max_retries: int = 10," \
        "max_retries: int = 2,"
    substituteInPlace pyproject.toml \
      --replace 'polars-lts-cpu' 'polars'
  '';

  sourceRoot = "multiqc";

  nativeBuildInputs = with python3Packages; [
    setuptools
    wheel
  ];

  dependencies = with python3Packages; [
    boto3
    click
    humanize
    importlib-metadata
    jinja2
    kaleido
    markdown
    natsort
    numpy
    packaging
    requests
    polars
    pillow
    plotly
    pyarrow
    pyyaml
    rich
    rich-click
    coloredlogs
    spectra
    pydantic
    tiktoken
    typeguard
    tqdm
    python-dotenv
    jsonschema
  ];

  optional-dependencies = {
    dev = with python3Packages; [
      pre-commit-hooks
      pdoc3
      pytest
      pytest-cov
      pytest-xdist
      syrupy
      pygithub
      mypy
      types-pyyaml
      types-tqdm
      types-requests
      types-markdown
      types-beautifulsoup4
      types-pillow
    ];
  };

  # Some tests run subprocess.run() with "multiqc"
  preCheck = ''
    chmod -R u+w ../test-data
    ln -s ../test-data .
  '';

  # Some tests run subprocess.run() with "ps"
  nativeCheckInputs =
    with python3Packages;
    [
      pytest-cov-stub
      pytest-xdist
      pytestCheckHook
      syrupy
      pygithub
      versionCheckHook
    ]
    ++ [
      procps
      addBinToPathHook
    ];

  versionCheckProgramArg = "--version";

  disabledTests =
    # On darwin, kaleido fails to starts
    lib.optionals (stdenv.hostPlatform.isDarwin) [
      "test_flat_plot"
    ];

  meta = {
    description = "Aggregates bioinformatics results from multiple samples into a unified report";
    longDescription = ''
      MultiQC is a tool to create a single report with interactive plots for multiple bioinformatics analyses across many samples.

      Reports are generated by scanning given directories for recognised log files. These are parsed and a single HTML report is generated summarising the statistics for all logs found. MultiQC reports can describe multiple analysis steps and large numbers of samples within a single plot, and multiple analysis tools making it ideal for routine fast quality control.
    '';
    homepage = "https://multiqc.info";
    changelog = "https://github.com/MultiQC/MultiQC/releases/tag/v${version}/";
    license = [ lib.licenses.gpl3Plus ];
    maintainers = [ lib.maintainers.apraga ];
    mainProgram = "multiqc";
    platforms = lib.platforms.unix;
  };

}
