#!/usr/bin/env python

# run with:
# $ nix run .\#vimPluginsUpdater
# format:
# $ nix run nixpkgs#python3Packages.ruff -- update.py
# type-check:
# $ nix run nixpkgs#python3Packages.mypy -- update.py
# linted:
# $ nix run nixpkgs#python3Packages.flake8 -- --ignore E501,E265,E402 update.py

# If you see `HTTP Error 429: too many requests` errors while running this
# script, refer to:
#
# https://github.com/NixOS/nixpkgs/blob/master/doc/languages-frameworks/vim.section.md#updating-plugins-in-nixpkgs-updating-plugins-in-nixpkgs
#
# (or the equivalent file /doc/languages-frameworks/vim.section.md
# from Nixpkgs master tree).
#

import inspect
import logging
import os
import textwrap
from pathlib import Path
from typing import List, Tuple

log = logging.getLogger("vim-updater")

ROOT = Path(os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe()))))
import importlib

import nixpkgs_plugin_update
from nixpkgs_plugin_update import PluginDesc, run_nix_expr

treesitter = importlib.import_module("nvim-treesitter.update")


HEADER = (
    "# GENERATED by ./pkgs/applications/editors/vim/plugins/utils/update.py. Do not edit!"
)

NIXPKGS_NVIMTREESITTER_FOLDER = "pkgs/applications/editors/vim/plugins/nvim-treesitter"


class VimEditor(nixpkgs_plugin_update.Editor):
    nvim_treesitter_updated = False

    def generate_nix(
        self, plugins: List[Tuple[PluginDesc, nixpkgs_plugin_update.Plugin]], outfile: str
    ):
        log.info("Generating nix code")
        log.debug("Loading nvim-treesitter revision from nix...")
        nvim_treesitter_rev = nixpkgs_plugin_update.run_nix_expr(
            "(import <localpkgs> { }).vimPlugins.nvim-treesitter.src.rev",
            self.nixpkgs,
            timeout=10,
        )

        GET_PLUGINS_LUA = """
        with import <localpkgs> {};
        lib.attrNames lua51Packages"""
        log.debug("Loading list of lua plugins...")
        luaPlugins = run_nix_expr(GET_PLUGINS_LUA, self.nixpkgs, timeout=30)

        def _isNeovimPlugin(plug: nixpkgs_plugin_update.Plugin) -> bool:
            """
            Whether it's a neovim-only plugin
            We can check if it's available in lua packages
            """
            if plug.normalized_name in luaPlugins:
                log.debug("%s is a neovim plugin", plug)
                return True
            return False

        with open(outfile, "w+") as f:
            log.debug("Writing to %s", outfile)
            f.write(HEADER)
            f.write(
                textwrap.dedent(
                    """
                {
                  lib,
                  buildVimPlugin,
                  buildNeovimPlugin,
                  fetchFromGitHub,
                }:

                final: prev: {
                """
                )
            )
            for pdesc, plugin in plugins:
                content = self.plugin2nix(pdesc, plugin, _isNeovimPlugin(plugin))
                f.write(content)
                if (
                    plugin.name == "nvim-treesitter"
                    and plugin.commit != nvim_treesitter_rev
                ):
                    self.nvim_treesitter_updated = True
            f.write("}\n")
        print(f"updated {outfile}")

    def plugin2nix(
        self, pdesc: PluginDesc, plugin: nixpkgs_plugin_update.Plugin, isNeovim: bool
    ) -> str:
        if isNeovim:
            raise RuntimeError(f"Plugin {plugin.name} is already packaged in `luaPackages`, please use that")
        repo = pdesc.repo

        content = f"  {plugin.normalized_name} = "
        src_nix = repo.as_nix(plugin)
        content += """{buildFn} {{
    pname = "{plugin.name}";
    version = "{plugin.version}";
    src = {src_nix};
    meta.homepage = "{repo.uri}";
    meta.hydraPlatforms = [ ];
  }};

""".format(
            buildFn="buildNeovimPlugin" if isNeovim else "buildVimPlugin",
            plugin=plugin,
            src_nix=src_nix,
            repo=repo,
        )
        log.debug(content)
        return content

    def update(self, args):
        nixpkgs_plugin_update.update_plugins(self, args)

        # TODO this should probably be skipped when running outside a nixpkgs checkout
        if self.nvim_treesitter_updated:
            print("updating nvim-treesitter grammars")
            generated = treesitter.update_grammars()
            treesitter_generated_nix_path = os.path.join(
                NIXPKGS_NVIMTREESITTER_FOLDER, "generated.nix"
            )
            open(os.path.join(args.nixpkgs, treesitter_generated_nix_path), "w").write(
                generated
            )

            if self.nixpkgs_repo:
                index = self.nixpkgs_repo.index
                for diff in index.diff(None):
                    if diff.a_path == treesitter_generated_nix_path:
                        msg = "vimPlugins.nvim-treesitter: update grammars"
                        print(f"committing to nixpkgs: {msg}")
                        index.add([treesitter_generated_nix_path])
                        index.commit(msg)
                        return
                print("no updates to nvim-treesitter grammars")


def main():
    global luaPlugins

    log.debug(f"Loading from {ROOT}/get-plugins.nix")
    with open(f"{ROOT}/get-plugins.nix") as f:
        GET_PLUGINS = f.read()
    editor = VimEditor(
        "vim", Path("pkgs/applications/editors/vim/plugins"), GET_PLUGINS
    )
    editor.run()


if __name__ == "__main__":
    main()
