{
  lib,
  stdenv,
  buildPackages,
  fetchurl,
  perl,
  libintl,
  bashNonInteractive,
  updateAutotoolsGnuConfigScriptsHook,
  gawk,
  freebsd,
  glibcLocales,
  libiconv,

  # we are a dependency of gcc, this simplifies bootstrapping
  interactive ? false,
  ncurses,
  procps,
  meta,
}:

{
  version,
  hash,
  patches ? [ ],
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

let
  inherit (lib)
    getBin
    getDev
    getLib
    optional
    optionals
    optionalString
    versionOlder
    ;
  crossBuildTools = stdenv.hostPlatform != stdenv.buildPlatform;
in

stdenv.mkDerivation {
  pname = "texinfo${optionalString interactive "-interactive"}";
  inherit version;

  src = fetchurl {
    url = "mirror://gnu/texinfo/texinfo-${version}.tar.xz";
    inherit hash;
  };

  patches = patches ++ optional crossBuildTools ./cross-tools-flags.patch;

  postPatch = ''
    patchShebangs tp/maintain/regenerate_commands_perl_info.pl
  '';

  env = {
    XFAIL_TESTS = toString (
      optionals stdenv.hostPlatform.isMusl [
        # musl does not support locales.
        "different_languages_gen_master_menu.sh"
        "test_scripts/formatting_documentlanguage_cmdline.sh"
        "test_scripts/layout_formatting_fr_info.sh"
        "test_scripts/layout_formatting_fr.sh"
        "test_scripts/layout_formatting_fr_icons.sh"
      ]
      ++ optionals (!stdenv.hostPlatform.isMusl && versionOlder version "7") [
        # Test is known to fail on various locales on texinfo-6.8:
        #   https://lists.gnu.org/r/bug-texinfo/2021-07/msg00012.html
        "test_scripts/layout_formatting_fr_icons.sh"
      ]
    );
  }
  // lib.optionalAttrs crossBuildTools {
    # ncurses is required to build `makedoc'
    # this feature is introduced by the ./cross-tools-flags.patch
    NATIVE_TOOLS_CFLAGS = "-I${getDev buildPackages.ncurses}/include";
    NATIVE_TOOLS_LDFLAGS = "-L${getLib buildPackages.ncurses}/lib";
  };

  strictDeps = true;
  enableParallelBuilding = true;

  # A native compiler is needed to build tools needed at build time
  depsBuildBuild = [
    buildPackages.stdenv.cc
    perl
  ];

  nativeBuildInputs = [ updateAutotoolsGnuConfigScriptsHook ];
  buildInputs = [
    bashNonInteractive
    libintl
  ]
  ++ optionals stdenv.hostPlatform.isSunOS [
    libiconv
    gawk
  ]
  ++ optional interactive ncurses;

  configureFlags = [
    "PERL=${buildPackages.perl}/bin/perl"
  ]
  # Perl XS modules are difficult to cross-compile and texinfo has pure Perl
  # fallbacks.
  # Also prevent the buildPlatform's awk being used in the texindex script
  ++ optionals crossBuildTools [
    "--enable-perl-xs=no"
    "TI_AWK=${getBin gawk}/bin/awk"
  ]
  ++ optionals (crossBuildTools && lib.versionAtLeast version "7.1") [
    "texinfo_cv_sys_iconv_converts_euc_cn=yes"
  ]
  ++ optional stdenv.hostPlatform.isSunOS "AWK=${gawk}/bin/awk";

  installFlags = [ "TEXMF=$(out)/texmf-dist" ];
  installTargets = [
    "install"
    "install-tex"
  ];

  nativeCheckInputs = [ procps ] ++ optionals stdenv.buildPlatform.isFreeBSD [ freebsd.locale ];
  checkInputs = optionals (lib.versionAtLeast version "7.2") [ glibcLocales ];

  doCheck = interactive && !stdenv.hostPlatform.isDarwin && !stdenv.hostPlatform.isSunOS; # flaky

  postFixup = optionalString crossBuildTools ''
    for f in "$out"/bin/{pod2texi,texi2any}; do
      substituteInPlace "$f" \
        --replace-fail ${buildPackages.perl}/bin/perl ${perl}/bin/perl
    done
  '';

  meta = meta // {
    branch = version;
  };
}
