#! /usr/bin/env nix-shell
#! nix-shell -i bash -p curl jq git gnused gnugrep -I nixpkgs=.
# shellcheck shell=bash
#
# SYNOPSIS
#
#   Update version constraints in hackage2nix config file from Stackage.
#
# DESCRIPTION
#
#   Fetches the latest snapshot of the configured Stackage solver which is
#   configured via the SOLVER (either LTS or Nightly) and VERSION variables in
#   the script.
#
#   VERSION is only applicable if SOLVER is LTS. SOLVER=LTS and VERSION=22
#   will cause update-stackage.sh to fetch the latest LTS-22.XX version.
#   If empty, the latest version of the solver is used.
#
#   If the configuration file has been updated, update-stackage.sh prints a
#   version difference to stdout, e.g. 23.11 -> 23.13. Otherwise, stdout remains
#   empty.
#
# EXIT STATUS
#
#   Always exit with zero (even if nothing changed) unless there was an error.

set -eu -o pipefail

if [[ "${1:-}" == "--do-commit" ]]; then
  echo "$0: --do-commit is no longer supported. Use update-package-set.sh instead."
  exit 100
fi

# Stackage solver to use, LTS or Nightly
# (should be capitalized like the display name)
SOLVER=LTS
# Stackage solver verson, if any. Use latest if empty
VERSION=
TMP_TEMPLATE=update-stackage.XXXXXXX
readonly SOLVER
readonly VERSION
readonly TMP_TEMPLATE

toLower() {
    printf "%s" "$1" | tr '[:upper:]' '[:lower:]'
}

tmpfile=$(mktemp "$TMP_TEMPLATE")
tmpfile_new=$(mktemp "$TMP_TEMPLATE")

stackage_config="pkgs/development/haskell-modules/configuration-hackage2nix/stackage.yaml"

trap 'rm "${tmpfile}" "${tmpfile_new}"' 0
touch "$tmpfile" "$tmpfile_new" # Creating files here so that trap creates no errors.

curl -L -s "https://stackage.org/$(toLower "$SOLVER")${VERSION:+-$VERSION}/cabal.config" >"$tmpfile"
old_version=$(grep '^# Stackage' $stackage_config | sed -e 's/.\+ \([A-Za-z]\+ [0-9.-]\+\)$/\1/g')
version="$SOLVER $(sed -rn "s/^--.*http:..(www.)?stackage.org.snapshot.$(toLower "$SOLVER")-//p" "$tmpfile")"

if [[ "$old_version" == "$version" ]]; then
   echo "No new stackage version" >&2
   exit 0 # Nothing to do
fi

echo "Updating Stackage from $old_version to $version." >&2

# Create a simple yaml version of the file.
sed -r \
    -e '/^--/d' \
    -e 's|^constraints:||' \
    -e 's|^ +|  - |' \
    -e 's|,$||' \
    -e '/^with-compiler:/d' \
    -e '/installed$/d' \
    -e '/^$/d' \
    < "${tmpfile}" | LC_ALL=C.UTF-8 sort --ignore-case >"${tmpfile_new}"

cat > $stackage_config << EOF
# Stackage $version
# This file is auto-generated by
# maintainers/scripts/haskell/update-stackage.sh
default-package-overrides:
EOF

# Drop restrictions on some tools where we always want the latest version.
sed -r \
    -e '/ cabal2nix /d' \
    -e '/ distribution-nixpkgs /d' \
    -e '/ jailbreak-cabal /d' \
    -e '/ language-nix /d' \
    -e '/ hackage-db /d' \
    -e '/ cabal-install /d' \
    -e '/ cabal-install-solver /d' \
    -e '/ lsp /d' \
    -e '/ lsp-types /d' \
    -e '/ lsp-test /d' \
    -e '/ hie-bios /d' \
    -e '/ ShellCheck /d' \
    -e '/ Agda /d' \
    -e '/ stack /d' \
    -e '/ git-annex /d' \
    < "${tmpfile_new}" >> $stackage_config
# Explanations:
# cabal2nix, distribution-nixpkgs, jailbreak-cabal, language-nix: These are our packages and we know what we are doing.
# lsp, lsp-types, lsp-test, hie-bios: These are tightly coupled to hls which is not in stackage. They have no rdeps in stackage.
# ShellCheck: latest version of command-line dev tool.
# Agda: The Agda community is fast-moving; we strive to always include the newest versions of Agda and the Agda packages in nixpkgs.

echo "$old_version -> $version"
