{
  lib,
  stdenv,
  fetchurl,
  autoreconfHook,
  updateAutotoolsGnuConfigScriptsHook,
  libintl,
  gettext,
  aclSupport ? lib.meta.availableOn stdenv.hostPlatform acl,
  acl,
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

stdenv.mkDerivation rec {
  pname = "gnutar";
  version = "1.35";

  src = fetchurl {
    url = "mirror://gnu/tar/tar-${version}.tar.xz";
    sha256 = "sha256-TWL/NzQux67XSFNTI5MMfPlKz3HDWRiCsmp+pQ8+3BY=";
  };

  # GNU tar fails to link libiconv even though the configure script detects it.
  # https://savannah.gnu.org/bugs/index.php?64441
  patches = [ ./link-libiconv.patch ];

  # gnutar tries to call into gettext between `fork` and `exec`,
  # which is not safe on darwin.
  # see http://article.gmane.org/gmane.os.macosx.fink.devel/21882
  postPatch = lib.optionalString stdenv.hostPlatform.isDarwin ''
    substituteInPlace src/system.c --replace '_(' 'N_('
  '';

  outputs = [
    "out"
    "info"
  ];

  nativeBuildInputs = [ autoreconfHook ];

  # Add libintl on Darwin specifically as it fails to link (or skip)
  # NLS on it's own:
  #  "_libintl_textdomain", referenced from:
  #    _main in tar.o
  #  ld: symbol(s) not found for architecture x86_64
  buildInputs =
    lib.optional aclSupport acl
    ++ lib.optional stdenv.hostPlatform.isDarwin libintl
    # gettext gets pulled in via autoreconfHook because strictDeps is not set,
    # and is linked against. Without this, it doesn't end up in HOST_PATH.
    # TODO: enable strictDeps, and either make this dependency explicit, or remove it
    ++ lib.optional stdenv.hostPlatform.isCygwin gettext;

  # May have some issues with root compilation because the bootstrap tool
  # cannot be used as a login shell for now.
  FORCE_UNSAFE_CONFIGURE = lib.optionalString (
    stdenv.hostPlatform.system == "armv7l-linux" || stdenv.hostPlatform.isSunOS
  ) "1";

  doCheck = false; # fails
  doInstallCheck = false; # fails

  meta = {
    description = "GNU implementation of the `tar' archiver";
    longDescription = ''
      The Tar program provides the ability to create tar archives, as
      well as various other kinds of manipulation.  For example, you
      can use Tar on previously created archives to extract files, to
      store additional files, or to update or list files which were
      already stored.

      Initially, tar archives were used to store files conveniently on
      magnetic tape.  The name "Tar" comes from this use; it stands
      for tape archiver.  Despite the utility's name, Tar can direct
      its output to available devices, files, or other programs (using
      pipes), it can even access remote devices or files (as
      archives).
    '';
    homepage = "https://www.gnu.org/software/tar/";

    license = lib.licenses.gpl3Plus;

    maintainers = with lib.maintainers; [ RossComputerGuy ];
    mainProgram = "tar";
    platforms = lib.platforms.all;

    priority = 10;
  };
}
