{
  lib,
  stdenv,
  fetchurl,
  mrustc,
  mrustc-minicargo,
  #llvm_12,
  libffi,
  cmake,
  perl,
  python3,
  zlib,
  libxml2,
  pkg-config,
  curl,
  which,
  time,
}:

let
  mrustcTargetVersion = "1.54";
  rustcVersion = "1.54.0";
  rustcSrc = fetchurl {
    url = "https://static.rust-lang.org/dist/rustc-${rustcVersion}-src.tar.gz";
    sha256 = "0xk9dhfff16caambmwij67zgshd8v9djw6ha0fnnanlv7rii31dc";
  };
  rustcDir = "rustc-${rustcVersion}-src";
  outputDir = "output-${rustcVersion}";
in

stdenv.mkDerivation rec {
  pname = "mrustc-bootstrap";
  version = "${mrustc.version}_${rustcVersion}";

  inherit (mrustc) src;
  postUnpack = "tar -xf ${rustcSrc} -C source/";

  # the rust build system complains that nix alters the checksums
  dontFixLibtool = true;

  patches = [
    ./patches/0001-dont-download-rustc.patch
  ];

  postPatch = ''
    echo "applying patch ./rustc-${rustcVersion}-src.patch"
    patch -p0 -d ${rustcDir}/ < rustc-${rustcVersion}-src.patch
  '';

  # rustc unfortunately needs cmake to compile llvm-rt but doesn't
  # use it for the normal build. This disables cmake in Nix.
  dontUseCmakeConfigure = true;

  strictDeps = true;
  nativeBuildInputs = [
    cmake
    mrustc
    mrustc-minicargo
    perl
    pkg-config
    python3
    time
    which
  ];
  buildInputs = [
    # for rustc
    #llvm_12
    libffi
    zlib
    libxml2
    # for cargo
    curl
  ];

  makeFlags = [
    # Use shared mrustc/minicargo/llvm instead of rebuilding them
    "MRUSTC=${mrustc}/bin/mrustc"
    #"MINICARGO=${mrustc-minicargo}/bin/minicargo"  # FIXME: we need to rebuild minicargo locally so --manifest-overrides is applied
    #"LLVM_CONFIG=${llvm_12.dev}/bin/llvm-config"
    "RUSTC_TARGET=${stdenv.targetPlatform.rust.rustcTarget}"
  ];

  buildPhase = ''
    runHook preBuild

    local flagsArray=(
      PARLEVEL=$NIX_BUILD_CORES
      ${toString makeFlags}
    )

    touch ${rustcDir}/dl-version
    export OUTDIR_SUF=-${rustcVersion}
    export RUSTC_VERSION=${rustcVersion}
    export MRUSTC_TARGET_VER=${mrustcTargetVersion}
    export MRUSTC_PATH=${mrustc}/bin/mrustc

    echo minicargo.mk: libs
    make -f minicargo.mk "''${flagsArray[@]}" LIBS

    echo test
    make "''${flagsArray[@]}" test

    # disabled because it expects ./bin/mrustc
    #echo local_tests
    #make "''${flagsArray[@]}" local_tests

    echo minicargo.mk: rustc
    make -f minicargo.mk "''${flagsArray[@]}" ${outputDir}/rustc

    echo minicargo.mk: cargo
    make -f minicargo.mk "''${flagsArray[@]}" ${outputDir}/cargo

    echo run_rustc
    make -C run_rustc "''${flagsArray[@]}"

    unset flagsArray

    runHook postBuild
  '';

  doCheck = true;
  checkPhase = ''
    runHook preCheck
    run_rustc/${outputDir}/prefix/bin/hello_world | grep "hello, world"
    runHook postCheck
  '';

  installPhase = ''
    runHook preInstall
    mkdir -p $out/bin/ $out/lib/
    cp run_rustc/${outputDir}/prefix/bin/cargo $out/bin/cargo
    cp run_rustc/${outputDir}/prefix/bin/rustc_binary $out/bin/rustc

    cp -r run_rustc/${outputDir}/prefix/lib/* $out/lib/
    cp $out/lib/rustlib/${stdenv.targetPlatform.rust.rustcTarget}/lib/*.so $out/lib/
    runHook postInstall
  '';

  meta = {
    inherit (src.meta) homepage;
    description = "Minimal build of Rust";
    longDescription = ''
      A minimal build of Rust, built from source using mrustc.
      This is useful for bootstrapping the main Rust compiler without
      an initial binary toolchain download.
    '';
    maintainers = with lib.maintainers; [
      progval
      r-burns
    ];
    license = with lib.licenses; [
      mit
      asl20
    ];
    platforms = [ "x86_64-linux" ];
    # rustc 1.54 only supports LLVM 12, which was removed from Nixpkgs.
    # mrustc can bootstrap up to rustc 1.74, which supported LLVM 17,
    # which has also been removed.
    #
    # 1.74 also shipped with the Cranelift backend, so perhaps that
    # could be used instead? Alternatively, it may be possible to
    # backport the upstream patches to support LLVM 18 to 1.74.
    # Assuming LLVM 18 is still in Nixpkgs by the time you read this
    # comment, anyway. But if not, then maybe mrustc has been updated
    # to support newer rustc versions? Hope springs eternal.
    #
    # (Note that you still have to “draw the rest of the owl” to
    # bootstrap the chain of rustc versions between this bootstrap
    # and the version currently used in Nixpkgs, anyway, so this was
    # already not useful for bootstrapping a Rust compiler for use with
    # Nixpkgs without a lot of additional work. See Guix’s Rust
    # bootstrap chain, or the non‐Rust minimal bootstrap in Guix and
    # Nixpkgs, for inspiration.)
    broken = true;
  };
}
