{
  lib,
  stdenv,
  fetchFromGitHub,
  cacert,
  autoconf,
  automake,
  libtool,
  cmake,
  pkg-config,
  python3,
  macdylibbundler,
  makeWrapper,
  darwin,
  codec2,
  libpulseaudio,
  libsamplerate,
  libsndfile,
  lpcnet,
  portaudio,
  speexdsp,
  hamlib_4,
  wxGTK32,
  dbus,
  apple-sdk_15,
  nix-update-script,
}:

let
  codec2' = codec2.override { freedvSupport = true; };
  ebur128Src = fetchFromGitHub {
    owner = "jiixyj";
    repo = "libebur128";
    rev = "v1.2.6";
    hash = "sha256-UKO2k+kKH/dwt2xfaYMrH/GXjEkIrnxh1kGG/3P5d3Y=";
  };
  opusSrc = fetchFromGitHub {
    owner = "xiph";
    repo = "opus";
    rev = "940d4e5af64351ca8ba8390df3f555484c567fbb";
    postFetch = ''
      cd $out
      export NIX_SSL_CERT_FILE=${cacert}/etc/ssl/certs/ca-bundle.crt
      export SSL_CERT_FILE=$NIX_SSL_CERT_FILE
      dnn/download_model.sh "4ed9445b96698bad25d852e912b41495ddfa30c8dbc8a55f9cde5826ed793453"
      substituteInPlace autogen.sh \
        --replace-fail 'dnn/download_model.sh "4ed9445b96698bad25d852e912b41495ddfa30c8dbc8a55f9cde5826ed793453"' ""
    '';
    hash = "sha256-P84gjnuiQQBVBExJBY3sUbwo00lXY6HB+AMpx/oovRg=";
  };
  radaeSrc = fetchFromGitHub {
    owner = "drowe67";
    repo = "radae";
    rev = "5d640a028ab2b8e4ff23ed7136caee396cdcb844";
    # upstream repository archive fetching is broken
    forceFetchGit = true;
    hash = "sha256-+Sd+FWycEJabT3RN/zyKXS2Xzr060/ekYdzg6s1gQcM=";
  };
  radeInteg = fetchFromGitHub {
    owner = "drowe67";
    repo = "radae";
    rev = "7bd3ae2401fcba58e314755576a2940085835312";
    hash = "sha256-WVYKvttiNh6uEzw0b27winyDfzzGkEEhYq7DIwfZW74=";
  };
in
stdenv.mkDerivation (finalAttrs: {
  pname = "freedv";
  version = "2.2.1";

  src = fetchFromGitHub {
    owner = "drowe67";
    repo = "freedv-gui";
    tag = "v${finalAttrs.version}";
    hash = "sha256-7SOGz2+MzAkXd5JDKasSJcKVXcnuYk+C0S9N/NPRfOM=";
  };

  patches = [
    ./no-framework.patch
  ];

  postPatch = ''
    cp -R ${ebur128Src} ebur128
    cp -R ${radaeSrc} radae
    cp -R ${radeInteg} rade_integ
    chmod -R u+w ebur128 radae rade_integ
    substituteInPlace cmake/BuildEbur128.cmake \
      --replace-fail "GIT_REPOSITORY https://github.com/jiixyj/libebur128.git" "URL $(realpath ebur128)" \
      --replace-fail 'GIT_TAG "v''${EBUR128_VERSION}"' "" \
      --replace-fail "git apply" "patch -p1 <"
    substituteInPlace cmake/BuildRADE.cmake \
      --replace-fail "https://github.com/xiph/opus/archive/940d4e5af64351ca8ba8390df3f555484c567fbb.zip" "${opusSrc}" \
      --replace-fail "GIT_REPOSITORY https://github.com/drowe67/radae.git" "URL $(realpath radae)" \
      --replace-fail "GIT_TAG main" ""
    substituteInPlace cmake/BuildRADEForIntegrations.cmake \
      --replace-fail "https://github.com/xiph/opus/archive/940d4e5af64351ca8ba8390df3f555484c567fbb.zip" "${opusSrc}" \
      --replace-fail "GIT_REPOSITORY https://github.com/drowe67/radae.git" "URL $(realpath rade_integ)" \
      --replace-fail "GIT_TAG ms-disable-python-gc" ""
    patchShebangs test/test_*.sh
    substituteInPlace cmake/CheckGit.cmake \
      --replace-fail "git describe --abbrev=4 --always HEAD" "echo v${finalAttrs.version}"
  ''
  + lib.optionalString stdenv.hostPlatform.isDarwin ''
    substituteInPlace CMakeLists.txt \
      --replace-fail "-Wl,-ld_classic" ""
    substituteInPlace src/CMakeLists.txt \
      --replace-fail "\''${CMAKE_SOURCE_DIR}/macdylibbundler/dylibbundler" "dylibbundler"
    sed -i "/codesign/d;/hdiutil/d" src/CMakeLists.txt
  '';

  nativeBuildInputs = [
    autoconf
    automake
    libtool
    cmake
    pkg-config
    python3
  ]
  ++ lib.optionals stdenv.hostPlatform.isDarwin [
    (macdylibbundler.overrideAttrs {
      # incompatible with darwin.sigtool in Nixpkgs
      postPatch = ''
        substituteInPlace src/Utils.cpp \
          --replace-fail "--deep --preserve-metadata=entitlements,requirements,flags,runtime" ""
      '';
    })
    makeWrapper
    darwin.autoSignDarwinBinariesHook
    darwin.sigtool
  ];

  buildInputs = [
    codec2'
    libsamplerate
    libsndfile
    lpcnet
    speexdsp
    hamlib_4
    wxGTK32
    python3.pkgs.numpy
  ]
  ++ (
    if stdenv.hostPlatform.isLinux then
      [
        libpulseaudio
        dbus
      ]
    else if stdenv.hostPlatform.isDarwin then
      [
        apple-sdk_15
      ]
    else
      [
        portaudio
      ]
  );

  cmakeFlags = [
    (lib.cmakeBool "USE_INTERNAL_CODEC2" false)
    (lib.cmakeBool "USE_STATIC_DEPS" false)
    (lib.cmakeBool "UNITTEST" true)
    (lib.cmakeBool "USE_NATIVE_AUDIO" (with stdenv.hostPlatform; isLinux || isDarwin))
  ];

  env.NIX_CFLAGS_COMPILE = "-I${codec2'.src}/src";

  doCheck = false;

  postInstall = ''
    install -Dm755 rade_build/src/librade.* -t $out/lib
  ''
  + lib.optionalString stdenv.hostPlatform.isDarwin ''
    mkdir -p $out/Applications
    mv $out/bin/FreeDV.app $out/Applications
    makeWrapper $out/Applications/FreeDV.app/Contents/MacOS/FreeDV $out/bin/freedv
  '';

  passthru.updateScript = nix-update-script {
    extraArgs = [
      # avoid pre‐releases
      "--version-regex"
      ''^v(\d\.\d\.\d(\.\d)?)$''
    ];
  };

  meta = {
    homepage = "https://freedv.org/";
    description = "Digital voice for HF radio";
    license = lib.licenses.lgpl21Only;
    maintainers = with lib.maintainers; [
      mvs
      wegank
    ];
    platforms = lib.platforms.unix;
    mainProgram = "freedv";
  };
})
