{
  lib,
  stdenv,
  fetchurl,
  fetchpatch,
  openssl,
  openldap,
  libkrb5,
  db,
  gettext,
  pam,
  libxcrypt,
  fixDarwinDylibNames,
  autoreconfHook,
  enableLdap ? false,
  buildPackages,
  pruneLibtoolFiles,
  nixosTests,
}:

stdenv.mkDerivation rec {
  pname = "cyrus-sasl";
  version = "2.1.28";

  src = fetchurl {
    urls = [
      "https://github.com/cyrusimap/${pname}/releases/download/${pname}-${version}/${pname}-${version}.tar.gz"
      "http://www.cyrusimap.org/releases/${pname}-${version}.tar.gz"
      "http://www.cyrusimap.org/releases/old/${pname}-${version}.tar.gz"
    ];
    sha256 = "sha256-fM/Gq9Ae1nwaCSSzU+Um8bdmsh9C1FYu5jWo6/xbs4w=";
  };

  patches = [
    # Fix cross-compilation
    ./cyrus-sasl-ac-try-run-fix.patch
    # make compatible with openssl3. can probably be dropped with any release after 2.1.28
    (fetchpatch {
      url = "https://github.com/cyrusimap/cyrus-sasl/compare/cb549ef71c5bb646fe583697ebdcaba93267a237...dfaa62392e7caecc6ecf0097b4d73738ec4fc0a8.patch";
      hash = "sha256-pc0cZqj1QoxDqgd/j/5q3vWONEPrTm4Pr6MzHlfjRCc=";
    })
    # Fix build with gcc15
    (fetchpatch {
      url = "https://src.fedoraproject.org/rpms/cyrus-sasl/raw/388b80c6a8f93667587b4ac2e7992d0aa1c431f9/f/cyrus-sasl-2.1.28-gcc15.patch";
      hash = "sha256-AfSQXFtVh0IHG8Uw9nWMWlkQnyaX3ZMsdZLd7hTru7Q=";
    })
  ];

  outputs = [
    "bin"
    "dev"
    "out"
    "man"
    "devdoc"
  ];

  depsBuildBuild = [ buildPackages.stdenv.cc ];
  nativeBuildInputs = [
    autoreconfHook
    pruneLibtoolFiles
  ]
  ++ lib.optional stdenv.hostPlatform.isDarwin fixDarwinDylibNames;
  buildInputs = [
    openssl
    db
    gettext
    libkrb5
    libxcrypt
  ]
  ++ lib.optional enableLdap openldap
  ++ lib.optional (lib.meta.availableOn stdenv.hostPlatform pam) pam;

  configureFlags = [
    "--with-openssl=${openssl.dev}"
    "--with-plugindir=${placeholder "out"}/lib/sasl2"
    "--with-saslauthd=/run/saslauthd"
    "--enable-login"
    "--enable-shared"
  ]
  ++ lib.optional enableLdap "--with-ldap=${openldap.dev}"
  ++ lib.optionals (stdenv.targetPlatform.useLLVM or false) [
    "--disable-sample"
    "CFLAGS=-DTIME_WITH_SYS_TIME"
  ];

  env = lib.optionalAttrs stdenv.cc.isGNU {
    NIX_CFLAGS_COMPILE = "-Wno-error=implicit-function-declaration";
  };

  installFlags = lib.optionals stdenv.hostPlatform.isDarwin [
    "framedir=$(out)/Library/Frameworks/SASL2.framework"
  ];

  passthru.tests = {
    inherit (nixosTests) parsedmarc postfix;
  };

  meta = {
    homepage = "https://www.cyrusimap.org/sasl";
    description = "Library for adding authentication support to connection-based protocols";
    platforms = lib.platforms.unix;
    license = lib.licenses.bsdOriginal;
  };
}
