{
  lib,
  stdenv,
  fetchurl,
  autoreconfHook,
  pkg-config,
  pruneLibtoolFiles,
  flex,
  bison,
  libmnl,
  libnetfilter_conntrack,
  libnfnetlink,
  libnftnl,
  libpcap,
  bash,
  bashNonInteractive,
  nftablesCompat ? true,
  gitUpdater,

  # For tests
  vmTools,
  python3,
  util-linux,
  nftables,
  strace,
  iana-etc,
  shadow,
  iproute2,
  iputils,
}:

let
  version = "1.8.11";
  pname = "iptables";
in

stdenv.mkDerivation (finalAttrs: {
  inherit pname version;

  __structuredAttrs = true;

  src = fetchurl {
    url = "https://www.netfilter.org/projects/${pname}/files/${pname}-${version}.tar.xz";
    sha256 = "2HMD1V74ySvK1N0/l4sm0nIBNkKwKUJXdfW60QCf57I=";
  };

  outputs = [
    "out"
    "lib"
    "dev"
    "man"
  ];

  strictDeps = true;

  nativeBuildInputs = [
    autoreconfHook
    pkg-config
    pruneLibtoolFiles
    flex
    bison
  ];

  buildInputs = [
    libmnl
    libnetfilter_conntrack
    libnfnetlink
    libnftnl
    libpcap
    bash
  ];

  configureFlags = [
    "--enable-bpf-compiler"
    "--enable-devel"
    "--enable-libipq"
    "--enable-nfsynproxy"
    "--enable-shared"
  ]
  ++ lib.optional (!nftablesCompat) "--disable-nftables";

  enableParallelBuilding = true;

  postInstall = lib.optionalString nftablesCompat ''
    rm $out/sbin/{iptables,iptables-restore,iptables-save,ip6tables,ip6tables-restore,ip6tables-save}
    ln -sv xtables-nft-multi $out/bin/iptables
    ln -sv xtables-nft-multi $out/bin/iptables-restore
    ln -sv xtables-nft-multi $out/bin/iptables-save
    ln -sv xtables-nft-multi $out/bin/ip6tables
    ln -sv xtables-nft-multi $out/bin/ip6tables-restore
    ln -sv xtables-nft-multi $out/bin/ip6tables-save
  '';

  outputChecks.lib.disallowedRequisites = [
    bash
    bashNonInteractive
  ];

  passthru = {
    updateScript = gitUpdater {
      url = "https://git.netfilter.org/iptables";
      rev-prefix = "v";
    };

    # Tests are run in a VM because they require access to the kernel (to modify rule chains)
    tests.withCheck = vmTools.runInLinuxVM (
      finalAttrs.finalPackage.overrideAttrs (_: {
        memSize = 4096;
        nativeCheckInputs = [
          python3
          util-linux
          nftables
          strace
          iana-etc
          shadow
          iproute2
          iputils
        ];

        doCheck = true;

        preCheck = ''
          # Tests require /etc/{ethertypes,protocols,services}
          cp etc/ethertypes /etc/ethertypes
          ln -s ${iana-etc}/protocols /etc/protocols
          ln -s ${iana-etc}/services /etc/services

          # Some tests specifically require a root group with GID 0
          groupadd -g 0 root

          # Set up for "unprivileged" test (it tries to runuser -u nobody)
          groupadd -g 1000 nogroup
          useradd nobody -u 1000 -g nogroup -d /var/empty
          mkdir -p /etc/pam.d
          echo 'auth sufficient pam_permit.so' >> /etc/pam.d/runuser
          echo 'account required pam_permit.so' >> /etc/pam.d/runuser
          echo 'password required pam_permit.so' >> /etc/pam.d/runuser
          echo 'session required pam_permit.so' >> /etc/pam.d/runuser

          # /etc/protocols has an entry for 141/wesp now, which makes three tests fail. Fix the expected output
          # TODO(balsoft): see if this should be upstreamed
          sed -i -e 's/protocol 141/protocol wesp/' -e 's/l4proto 141/l4proto wesp/' -e 's/!= 141/!= wesp/' extensions/generic.txlate
          # Not sure what causes these failures. Just disable the tests for now.
          # FIXME(balsoft): see if this is fixed in a future release
          sed -i -e '/^monitorcheck \w*tables -X [^ ]*$/d' iptables/tests/shell/testcases/nft-only/0012-xtables-monitor_0

          ${lib.optionalString (stdenv.system == "aarch64-linux") ''
            # All SECMARK-related tests fail on aarch64 for some reason
            rm extensions/*SECMARK.t
          ''}

          patchShebangs xlate-test.py iptables-test.py iptables/tests
        '';

        # Save some resources by not installing anything
        outputs = [ "out" ];
        postCheck = ''
          touch "$out"
        '';

        dontInstall = true;
        dontFixup = true;
      })
    );
  };

  meta = {
    description = "Program to configure the Linux IP packet filtering ruleset";
    homepage = "https://www.netfilter.org/projects/iptables/index.html";
    platforms = lib.platforms.linux;
    mainProgram = "iptables";
    maintainers = with lib.maintainers; [ fpletz ];
    license = lib.licenses.gpl2Plus;
    downloadPage = "https://www.netfilter.org/projects/iptables/files/";
  };
})
