{
  autoreconfHook,
  fetchFromGitHub,
  fetchpatch,
  lib,
  libtraceevent,
  nix-update-script,
  nixosTests,
  perl,
  pkg-config,
  sqlite,
  stdenv,
  # Options
  enableDmidecode ? stdenv.hostPlatform.isx86_64,
  dmidecode,
}:

stdenv.mkDerivation (finalAttrs: {
  pname = "rasdaemon";
  version = "0.8.3";

  src = fetchFromGitHub {
    owner = "mchehab";
    repo = "rasdaemon";
    tag = "v${finalAttrs.version}";
    hash = "sha256-SpMNkeJkjaWteWsIScRnzNILf+PtVu1sX9e6ctwm3G0=";
  };

  patches = [
    # https://github.com/mchehab/rasdaemon/pull/212
    (fetchpatch {
      name = "fix_buffer_overflow_in_add_event_handler_read.patch";
      url = "https://github.com/mchehab/rasdaemon/commit/46bed1b6845bcb560d760b4cacea7df67cd6d1fd.patch";
      hash = "sha256-5T5U2i0i/7MpHzqpPq6mn2ghSUj9O6BzY11VcySgCMo=";
    })
  ];

  strictDeps = true;

  enableParallelBuilding = true;

  nativeBuildInputs = [
    autoreconfHook
    pkg-config
  ];

  buildInputs = [
    libtraceevent
    (perl.withPackages (
      ps: with ps; [
        DBDSQLite
      ]
    ))
    sqlite
  ]
  ++ lib.optionals enableDmidecode [
    dmidecode
  ];

  configureFlags = [
    "--sysconfdir=/etc"
    "--localstatedir=/var"
    "--enable-all"
  ];

  # The installation attempts to create the following directories:
  # /var/lib/rasdaemon
  #   location of the RAS event log generated by rasdaemon -r
  # /etc/ras/dimm_labels.d
  #   location of the DIMM labels generated by ras-mc-ctl
  # /etc/sysconfig/rasdaemon
  #   location of rasdaemon config file, currently only used for CE PFA config

  # these are optional (for logging, DIMM label storage and user config)
  # /var/lib/rasdaemon should be created by the NixOS module
  # /etc/ras/dimm_labels.d should probably be generated,
  # from user supplied content, in the NixOS module
  # /etc/sysconfig/rasdaemon should be generated if there is user supplied content
  # and default to $out/etc/sysconfig/rasdaemon which should hold the supplied default

  # therefore, stripping these from the generated Makefile
  # (needed in the config flags because those set where the tools look for these)

  # easy way out, ends up installing /nix/store/...rasdaemon/bin in $out

  postPatch = ''
    patchShebangs contrib/
  '';

  preConfigure = ''
    substituteInPlace Makefile.am \
      --replace-fail '"$(DESTDIR)@sysconfdir@/ras/dimm_labels.d"' '"$(prefix)@sysconfdir@/ras/dimm_labels.d"' \
      --replace-fail '"$(DESTDIR)@SYSCONFDEFDIR@/rasdaemon"' '"$(prefix)@SYSCONFDEFDIR@/rasdaemon"' \
      --replace-fail '"$(DESTDIR)@sysconfdir@/ras/triggers' '"$(prefix)@sysconfdir@/ras/triggers'
  '';

  outputs = [
    "out"
    "dev"
    "man"
    "inject"
  ];

  postInstall = ''
    install -Dm 0755 contrib/edac-fake-inject $inject/bin/edac-fake-inject
    install -Dm 0755 contrib/edac-tests $inject/bin/edac-tests
  '';

  postFixup = lib.optionalString enableDmidecode ''
    substituteInPlace $out/bin/ras-mc-ctl \
      --replace-fail 'find_prog ("dmidecode")' '"${dmidecode}/bin/dmidecode"'
  '';

  passthru.tests = nixosTests.rasdaemon;

  passthru.updateScript = nix-update-script { };

  meta = {
    description = ''
      A Reliability, Availability and Serviceability (RAS) logging tool using EDAC kernel tracing events
    '';
    longDescription = ''
      Rasdaemon is a RAS (Reliability, Availability and Serviceability) logging
      tool. It records memory errors, using the EDAC tracing events. EDAC is a
      Linux kernel subsystem with handles detection of ECC errors from memory
      controllers for most chipsets on i386 and x86_64 architectures. EDAC
      drivers for other architectures like arm also exists.
    '';
    homepage = "https://github.com/mchehab/rasdaemon";
    license = lib.licenses.gpl2Plus;
    platforms = lib.platforms.linux;
    changelog = "${finalAttrs.meta.homepage}/releases/tag/v${finalAttrs.version}";
    maintainers = [ ];
  };
})
