{
  lib,
  stdenv,
  fetchurl,

  # test suite depends on dejagnu which cannot be used during bootstrapping
  # dejagnu also requires tcl which can't be built statically at the moment
  doCheck ? !(stdenv.hostPlatform.isStatic),
  dejagnu,
  nix-update-script,
  testers,
}:

stdenv.mkDerivation (finalAttrs: {
  pname = "libffi";
  version = "3.5.2";

  src = fetchurl {
    url =
      with finalAttrs;
      "https://github.com/libffi/libffi/releases/download/v${version}/${pname}-${version}.tar.gz";
    hash = "sha256-86MIKiOzfCk6T80QUxR7Nx8v+R+n6hsqUuM1Z2usgtw=";
  };

  # Note: this package is used for bootstrapping fetchurl, and thus
  # cannot use fetchpatch! All mutable patches (generated by GitHub or
  # cgit) that are needed here should be included directly in Nixpkgs as
  # files.
  patches = [
    # Threading tests need to be linked against pthread
    # See: https://github.com/libffi/libffi/pull/944
    ./freebsd-tsan-pthread.patch
  ];

  strictDeps = true;
  outputs = [
    "out"
    "dev"
    "man"
    "info"
  ];

  enableParallelBuilding = true;

  configurePlatforms = [
    "build"
    "host"
  ];

  configureFlags = [
    "--with-gcc-arch=generic" # no detection of -march= or -mtune=
    "--enable-pax_emutramp"
  ];

  preCheck = ''
    # The tests use -O0 which is not compatible with -D_FORTIFY_SOURCE.
    NIX_HARDENING_ENABLE=''${NIX_HARDENING_ENABLE/fortify3/}
    NIX_HARDENING_ENABLE=''${NIX_HARDENING_ENABLE/fortify/}
  '';

  dontStrip = stdenv.hostPlatform != stdenv.buildPlatform; # Don't run the native `strip' when cross-compiling.

  inherit doCheck;

  nativeCheckInputs = [ dejagnu ];

  passthru = {
    updateScript = nix-update-script { };
    tests = {
      pkg-config = testers.hasPkgConfigModules {
        package = finalAttrs.finalPackage;
      };
    };
  };

  meta = with lib; {
    description = "Foreign function call interface library";
    longDescription = ''
      The libffi library provides a portable, high level programming
      interface to various calling conventions.  This allows a
      programmer to call any function specified by a call interface
      description at run-time.

      FFI stands for Foreign Function Interface.  A foreign function
      interface is the popular name for the interface that allows code
      written in one language to call code written in another
      language.  The libffi library really only provides the lowest,
      machine dependent layer of a fully featured foreign function
      interface.  A layer must exist above libffi that handles type
      conversions for values passed between the two languages.
    '';
    homepage = "http://sourceware.org/libffi/";
    license = licenses.mit;
    maintainers = with maintainers; [ matthewbauer ];
    platforms = platforms.all;
    pkgConfigModules = [ "libffi" ];
  };
})
