{
  lib,
  stdenv,
  updateAutotoolsGnuConfigScriptsHook,
  glibcLocales,
  fetchurl,
  pcre2,
  libiconv,
  perl,
  runtimeShellPackage,
}:

# Note: this package is used for bootstrapping fetchurl, and thus
# cannot use fetchpatch! All mutable patches (generated by GitHub or
# cgit) that are needed here should be included directly in Nixpkgs as
# files.

let
  version = "3.12";
in

stdenv.mkDerivation {
  pname = "gnugrep";
  inherit version;

  src = fetchurl {
    url = "mirror://gnu/grep/grep-${version}.tar.xz";
    hash = "sha256-JkmyfA6Q5jLq3NdXvgbG6aT0jZQd5R58D4P/dkCKB7k=";
  };

  patches = [
    # Fixes test-float-h failure on ppc64 with C23
    # https://lists.gnu.org/archive/html/bug-gnulib/2025-07/msg00021.html
    # Multiple upstream commits squashed with adjustments, see header
    ./gnulib-float-h-tests-port-to-C23-PowerPC-GCC.patch
  ];

  # Some gnulib tests fail
  # - on Musl: https://github.com/NixOS/nixpkgs/pull/228714
  # - on x86_64-darwin: https://github.com/NixOS/nixpkgs/pull/228714#issuecomment-1576826330
  postPatch =
    if stdenv.hostPlatform.isMusl || (stdenv.hostPlatform.isDarwin && stdenv.hostPlatform.isx86_64) then
      ''
        sed -i 's:gnulib-tests::g' Makefile.in
      ''
    else
      null;

  nativeCheckInputs = [
    perl
    glibcLocales
  ];
  outputs = [
    "out"
    "info"
  ]; # the man pages are rather small

  nativeBuildInputs = [ updateAutotoolsGnuConfigScriptsHook ];
  buildInputs = [
    pcre2
    libiconv
  ]
  ++ lib.optional (!stdenv.hostPlatform.isWindows) runtimeShellPackage;

  # cygwin: FAIL: multibyte-white-space
  # freebsd: FAIL mb-non-UTF8-performance
  # x86_64-darwin: fails 'stack-overflow' tests on Rosetta 2 emulator
  # aarch32: fails 'stack-overflow' when run on qemu under x86_64
  doCheck =
    !stdenv.hostPlatform.isCygwin
    && !stdenv.hostPlatform.isFreeBSD
    && !(stdenv.hostPlatform.isDarwin && stdenv.hostPlatform.isx86_64)
    && !stdenv.buildPlatform.isRiscV64
    && !stdenv.hostPlatform.isAarch32;

  # On macOS, force use of mkdir -p, since Grep's fallback
  # (./install-sh) is broken.
  preConfigure = ''
    export MKDIR_P="mkdir -p"
  '';

  enableParallelBuilding = true;

  # Fix reference to sh in bootstrap-tools, and invoke grep via
  # absolute path rather than looking at argv[0].
  postInstall = ''
    rm $out/bin/egrep $out/bin/fgrep
    echo "#! /bin/sh" > $out/bin/egrep
    echo "exec $out/bin/grep -E \"\$@\"" >> $out/bin/egrep
    echo "#! /bin/sh" > $out/bin/fgrep
    echo "exec $out/bin/grep -F \"\$@\"" >> $out/bin/fgrep
    chmod +x $out/bin/egrep $out/bin/fgrep
  '';

  env = lib.optionalAttrs stdenv.hostPlatform.isMinGW {
    NIX_CFLAGS_COMPILE = "-Wno-error=format-security";
  };

  meta = with lib; {
    homepage = "https://www.gnu.org/software/grep/";
    description = "GNU implementation of the Unix grep command";

    longDescription = ''
      The grep command searches one or more input files for lines
      containing a match to a specified pattern.  By default, grep
      prints the matching lines.
    '';

    license = licenses.gpl3Plus;

    maintainers = [
      maintainers.das_j
      maintainers.m00wl
    ];
    platforms = platforms.all;
    mainProgram = "grep";
  };

  passthru = {
    inherit pcre2;
  };
}
