{
  config,
  lib,
  pkgs,
  options,
  ...
}:
let
  cfg = config.services.ipfs-cluster;

  # secret is by envvar, not flag
  initFlags = toString [
    (lib.optionalString (cfg.initPeers != [ ]) "--peers")
    (lib.strings.concatStringsSep "," cfg.initPeers)
  ];
in
{
  options = {

    services.ipfs-cluster = {

      enable = lib.mkEnableOption "Pinset orchestration for IPFS - requires ipfs daemon to be useful";

      consensus = lib.mkOption {
        type = lib.types.enum [
          "raft"
          "crdt"
        ];
        description = "Consensus protocol - 'raft' or 'crdt'. <https://cluster.ipfs.io/documentation/guides/consensus/>";
      };

      dataDir = lib.mkOption {
        type = lib.types.str;
        default = "/var/lib/ipfs-cluster";
        description = "The data dir for ipfs-cluster.";
      };

      initPeers = lib.mkOption {
        type = lib.types.listOf lib.types.str;
        default = [ ];
        description = "Peer addresses to initialize with on first run.";
      };

      openSwarmPort = lib.mkOption {
        type = lib.types.bool;
        default = false;
        description = "Open swarm port, secured by the cluster secret. This does not expose the API or proxy. <https://cluster.ipfs.io/documentation/guides/security/>";
      };

      secretFile = lib.mkOption {
        type = lib.types.nullOr lib.types.path;
        default = null;
        description = ''
          File containing the cluster secret in the format of EnvironmentFile as described by
          {manpage}`systemd.exec(5)`. For example:
          <programlisting>
          CLUSTER_SECRET=<replaceable>...</replaceable>
          </programlisting>

          If null, a new secret will be generated on first run and stored in the data directory.
          A secret in the correct format can also be generated by: `openssl rand -hex 32`
        '';
      };
    };
  };

  config = lib.mkIf cfg.enable {
    assertions = [
      {
        assertion = cfg.enable -> config.services.kubo.enable;
        message = "ipfs-cluster requires ipfs - configure and enable services.kubo";
      }
    ];

    environment.systemPackages = [ pkgs.ipfs-cluster ];

    systemd.tmpfiles.rules = [
      "d '${cfg.dataDir}' - ${config.services.kubo.user} ${config.services.kubo.group} - -"
    ];

    systemd.services.ipfs-cluster-init = {
      path = [
        "/run/wrappers"
        pkgs.ipfs-cluster
      ];
      environment.IPFS_CLUSTER_PATH = cfg.dataDir;
      wantedBy = [ "default.target" ];

      serviceConfig = {
        ExecStart = [
          "${lib.getExe' pkgs.ipfs-cluster "ipfs-cluster-service"} init --consensus ${cfg.consensus} ${initFlags}"
        ];
        Type = "oneshot";
        RemainAfterExit = true;
        User = config.services.kubo.user;
        Group = config.services.kubo.group;
        EnvironmentFile = lib.mkIf (cfg.secretFile != null) cfg.secretFile;
      };
      # only run once (= when the data directory is empty)
      unitConfig.ConditionDirectoryNotEmpty = "!${cfg.dataDir}";
    };

    systemd.services.ipfs-cluster = {
      environment.IPFS_CLUSTER_PATH = cfg.dataDir;
      wantedBy = [ "multi-user.target" ];

      wants = [ "ipfs-cluster-init.service" ];
      after = [ "ipfs-cluster-init.service" ];

      serviceConfig = {
        Type = "notify";
        ExecStart = [ "${lib.getExe' pkgs.ipfs-cluster "ipfs-cluster-service"} daemon" ];
        User = config.services.kubo.user;
        Group = config.services.kubo.group;
      };
    };

    networking.firewall.allowedTCPPorts = lib.mkIf cfg.openSwarmPort [ 9096 ];
  };

  meta = {
    maintainers = with lib.maintainers; [
      sorki
    ];
  };
}
