# pkgs.ociTools {#sec-pkgs-ociTools}

`pkgs.ociTools` is a set of functions for creating runtime container bundles according to the [OCI runtime specification v1.0.0](https://github.com/opencontainers/runtime-spec/blob/v1.0.0/spec.md).
It makes no assumptions about the container runner you choose to use to run the created container.

The set of functions in `pkgs.ociTools` currently does not handle the [OCI image specification](https://github.com/opencontainers/image-spec).

At a high-level an OCI implementation would download an OCI Image then unpack that image into an OCI Runtime filesystem bundle.
At this point the OCI Runtime Bundle would be run by an OCI Runtime.
`pkgs.ociTools` provides utilities to create OCI Runtime bundles.

## buildContainer {#ssec-pkgs-ociTools-buildContainer}

This function creates an OCI runtime container (consisting of a `config.json` and a root filesystem directory) that runs a single command inside of it.
The nix store of the container will contain all referenced dependencies of the given command.

This function has an assumption that the container will run on POSIX platforms, and sets configurations (such as the user running the process or certain mounts) according to this assumption.
Because of this, a container built with `buildContainer` will not work on Windows or other non-POSIX platforms without modifications to the container configuration.
These modifications aren't supported by `buildContainer`.

For `linux` platforms, `buildContainer` also configures the following namespaces (see {manpage}`unshare(1)`) to isolate the OCI container from the global namespace:
PID, network, mount, IPC, and UTS.

Note that no user namespace is created, which means that you won't be able to run the container unless you are the `root` user.

### Inputs {#ssec-pkgs-ociTools-buildContainer-inputs}

`buildContainer` expects an argument with the following attributes:

`args` (List of String)

: Specifies a set of arguments to run inside the container.
  Any packages referenced by `args` will be made available inside the container.

`mounts` (Attribute Set; _optional_)

: Would specify additional mounts that the runtime must make available to the container.

  :::{.warning}
  As explained in [issue #290879](https://github.com/NixOS/nixpkgs/issues/290879), this attribute is currently ignored.
  :::

  :::{.note}
  `buildContainer` includes a minimal set of necessary filesystems to be mounted into the container, and this set can't be changed with the `mounts` attribute.
  :::

  _Default value:_ `{}`.

`readonly` (Boolean; _optional_)

: If `true`, sets the container's root filesystem as read-only.

  _Default value:_ `false`.

`os` **DEPRECATED**

: Specifies the operating system on which the container filesystem is based on.
  If specified, its value should follow the [OCI Image Configuration Specification](https://github.com/opencontainers/image-spec/blob/main/config.md#properties).
  According to the linked specification, all possible values for `$GOOS` in [the Go docs](https://go.dev/doc/install/source#environment) should be valid, but will commonly be one of `darwin` or `linux`.

  _Default value:_ `"linux"`.

`arch` **DEPRECATED**

: Used to specify the architecture for which the binaries in the container filesystem have been compiled.
  If specified, its value should follow the [OCI Image Configuration Specification](https://github.com/opencontainers/image-spec/blob/main/config.md#properties).
  According to the linked specification, all possible values for `$GOARCH` in [the Go docs](https://go.dev/doc/install/source#environment) should be valid, but will commonly be one of `386`, `amd64`, `arm`, or `arm64`.

  _Default value:_ `x86_64`.

### Examples {#ssec-pkgs-ociTools-buildContainer-examples}

::: {.example #ex-ociTools-buildContainer-bash}
# Creating an OCI runtime container that runs `bash`

This example uses `ociTools.buildContainer` to create a simple container that runs `bash`.

```nix
{
  ociTools,
  lib,
  bash,
}:
ociTools.buildContainer {
  args = [ (lib.getExe bash) ];

  readonly = false;
}
```

As an example of how to run the container generated by this package, we'll use `runc` to start the container.
Any other tool that supports OCI containers could be used instead.

```shell
$ nix-build
(some output removed for clarity)
/nix/store/7f9hgx0arvhzp2a3qphp28rxbn748l25-join

$ cd /nix/store/7f9hgx0arvhzp2a3qphp28rxbn748l25-join
$ nix-shell -p runc
[nix-shell:/nix/store/7f9hgx0arvhzp2a3qphp28rxbn748l25-join]$ sudo runc run ocitools-example
help
GNU bash, version 5.2.26(1)-release (x86_64-pc-linux-gnu)
(some output removed for clarity)
```
:::
