#!/usr/bin/env python
# format:
# $ nix run nixpkgs#python3Packages.ruff -- update.py
# type-check:
# $ nix run nixpkgs#python3Packages.mypy -- update.py
# linted:
# $ nix run nixpkgs#python3Packages.flake8 -- --ignore E501,E265,E402 update.py

import csv
import inspect
import logging
import os
import shutil
import subprocess
import tempfile
import textwrap
from dataclasses import dataclass
from multiprocessing.dummy import Pool
from pathlib import Path

import nixpkgs_plugin_update
from nixpkgs_plugin_update import FetchConfig, update_plugins


class ColoredFormatter(logging.Formatter):
    # Define color codes
    COLORS = {
        "DEBUG": "\033[94m",  # Blue
        "INFO": "\033[92m",  # Green
        "WARNING": "\033[93m",  # Yellow
        "ERROR": "\033[91m",  # Red
        "CRITICAL": "\033[95m",  # Magenta
    }
    RESET = "\033[0m"

    def format(self, record):
        log_color = self.COLORS.get(record.levelname, self.RESET)
        record.msg = f"{log_color}{record.msg}{self.RESET}"
        return super().format(record)


handler = logging.StreamHandler()
handler.setFormatter(ColoredFormatter("%(levelname)s: %(message)s"))
log = logging.getLogger()
log.handlers = [handler]

ROOT = Path(os.path.dirname(os.path.abspath(inspect.getfile(inspect.currentframe())))).parent.parent  # type: ignore

PKG_LIST = "maintainers/scripts/luarocks-packages.csv"
TMP_FILE = "$(mktemp)"
GENERATED_NIXFILE = "pkgs/development/lua-modules/generated-packages.nix"

HEADER = """/*
  {GENERATED_NIXFILE} is an auto-generated file -- DO NOT EDIT!
  Regenerate it with: nix run nixpkgs#luarocks-packages-updater
  You can customize the generated packages in pkgs/development/lua-modules/overrides.nix
*/
""".format(GENERATED_NIXFILE=GENERATED_NIXFILE)

FOOTER = """
}
# GENERATED - do not edit this file
"""


@dataclass
class LuaPlugin:
    name: str
    """Name of the plugin, as seen on luarocks.org"""
    rockspec: str
    """Full URI towards the rockspec"""
    ref: str | None
    """git reference (branch name/tag)"""
    version: str | None
    """Set it to pin a package """
    server: str | None
    """luarocks.org registers packages under different manifests.
    Its value can be 'http://luarocks.org/dev'
    """
    luaversion: str | None
    """lua version if a package is available only for a specific lua version"""
    maintainers: str | None
    """Optional string listing maintainers separated by spaces"""

    @property
    def normalized_name(self) -> str:
        return self.name.replace(".", "-")


# rename Editor to LangUpdate/ EcosystemUpdater
class LuaEditor(nixpkgs_plugin_update.Editor):
    def create_parser(self):
        parser = super().create_parser()
        parser.set_defaults(proc=1)
        return parser

    def get_current_plugins(self):
        return []

    def load_plugin_spec(self, input_file) -> list[LuaPlugin]:
        luaPackages = []
        csvfilename = input_file
        log.info("Loading package descriptions from %s", csvfilename)

        with open(csvfilename, newline="") as csvfile:
            reader = csv.DictReader(
                csvfile,
            )
            for row in reader:
                # name,server,version,luaversion,maintainers
                plugin = LuaPlugin(**row)
                luaPackages.append(plugin)
        return luaPackages

    def update(self, args):
        update_plugins(self, args)

    def generate_nix(self, results: list[tuple[LuaPlugin, str]], outfilename: str):
        with tempfile.NamedTemporaryFile("w+") as f:
            f.write(HEADER)
            header2 = textwrap.dedent(
            """
            {
              stdenv,
              lib,
              fetchurl,
              fetchgit,
              callPackage,
              ...
            }:
            final: prev: {
            """
            )
            f.write(header2)
            for plugin, nix_expr in results:
                f.write(f"{plugin.normalized_name} = {nix_expr}")
            f.write(FOOTER)
            f.flush()

            # if everything went fine, move the generated file to its destination
            # using copy since move doesn't work across disks
            shutil.copy(f.name, outfilename)

        print(f"updated {outfilename}")

        # Format the generated file with nix fmt
        subprocess.run(["nix", "fmt", outfilename], check=True)

    @property
    def attr_path(self):
        return "luaPackages"

    def get_update(
        self,
        input_file: str,
        output_file: str,
        config: FetchConfig,
        # TODO: implement support for adding/updating individual plugins
        to_update: list[str] | None,
    ):
        if to_update is not None:
            raise NotImplementedError("For now, lua updater doesn't support updating individual packages.")
        _prefetch = generate_pkg_nix

        def update() -> dict:
            plugin_specs = self.load_plugin_spec(input_file)
            sorted_plugin_specs = sorted(plugin_specs, key=lambda v: v.name.lower())

            try:
                pool = Pool(processes=config.proc)
                results = pool.map(_prefetch, sorted_plugin_specs)
            finally:
                pass

            successful_results = [(plug, nix_expr) for plug, nix_expr, error in results if nix_expr is not None]
            errors = [(plug, error) for plug, nix_expr, error in results if error is not None]

            self.generate_nix(successful_results, output_file)

            if errors:
                log.error("The following plugins failed to update:")
                for plug, error in errors:
                    log.error("%s: %s", plug.name, error)

            redirects = {}
            return redirects

        return update

    def rewrite_input(self, input_file: str, *args, **kwargs):
        # vim plugin reads the file before update but that shouldn't be our case
        # not implemented yet
        # fieldnames = ['name', 'server', 'version', 'luaversion', 'maintainers']
        # input_file = "toto.csv"
        # with open(input_file, newline='') as csvfile:
        #     writer = csv.DictWriter(csvfile, fieldnames=fieldnames)
        #     writer.writeheader()
        #     for row in reader:
        #         # name,server,version,luaversion,maintainers
        #         plugin = LuaPlugin(**row)
        #         luaPackages.append(plugin)
        pass


def generate_pkg_nix(plug: LuaPlugin):
    """
    Generate nix expression for a luarocks package
    Our cache key associates "p.name-p.version" to its rockspec
    """
    log.debug("Generating nix expression for %s", plug.name)

    try:
        cmd = ["luarocks", "nix"]

        if plug.maintainers:
            cmd.append(f"--maintainers={plug.maintainers}")

        if plug.rockspec != "":
            if plug.ref or plug.version:
                msg = "'version' and 'ref' will be ignored as the rockspec is hardcoded for package %s" % plug.name
                log.warning(msg)

            log.debug("Updating from rockspec %s", plug.rockspec)
            cmd.append(plug.rockspec)

        # update the plugin from luarocks
        else:
            cmd.append(plug.name)
            if plug.version and plug.version != "src":
                cmd.append(plug.version)

        if plug.server != "src" and plug.server:
            cmd.append(f"--only-server={plug.server}")

        if plug.luaversion:
            cmd.append(f"--lua-version={plug.luaversion}")
            luaver = plug.luaversion.replace(".", "")
            if luaver := os.getenv(f"LUA_{luaver}"):
                cmd.append(f"--lua-dir={luaver}")

        log.debug("running %s", " ".join(cmd))

        output = subprocess.check_output(cmd, text=True)
        ## FIXME: luarocks nix command output isn't formatted properly
        output = "callPackage(\n" + output.strip() + ") {};\n\n"
        return (plug, output, None)
    except subprocess.CalledProcessError as e:
        log.error("Failed to generate nix expression for %s: %s", plug.name, e)
        return (plug, None, str(e))


def main():
    editor = LuaEditor(
        "lua",
        ROOT,
        "",
        default_in=PKG_LIST,
        default_out=GENERATED_NIXFILE,
    )

    editor.run()


if __name__ == "__main__":
    main()
